<?php
// --- Data for the properties ---
// This array holds the information for each property, similar to the React component's state.
// Make sure the image paths in 'image' are correct relative to your project root.
$properties = [
    [
        'id' => '01',
        'title' => 'Enchanted Oasis Villa',
        'location' => 'Delhi, India',
        'image' => 'Assets/property1.jpg', // Update this path if needed
    ],
    [
        'id' => '02',
        'title' => 'Château Elegance',
        'location' => 'Manali, Himachal Pradesh',
        'image' => 'Assets/property2.jpg', // Update this path if needed
    ],
    [
        'id' => '03',
        'title' => 'The Glass House',
        'location' => 'Gangtok, Sikkim',
        'image' => 'Assets/property3.jpg', // Update this path if needed
    ],
];
?>

<!-- 
  SECTION: About Preview
  This section showcases featured properties with a unique circular wipe animation
  that is controlled by scrolling. It is a direct PHP/JS conversion of the original React component.
-->
<section 
    id="about-preview-section" 
    class="relative bg-slate-50 w-full min-h-screen flex flex-col lg:flex-row items-center justify-center py-16 px-4 lg:p-0 overflow-x-hidden"
>
    <!-- Right Content: Property Title and Location -->
    <div class="g-about-ui-element lg:absolute lg:right-12 lg:top-1/2 lg:-translate-y-1/2 text-center lg:text-right z-20 order-1 lg:order-3">
        <div class="relative h-28 w-full max-w-xs sm:max-w-sm md:max-w-md lg:w-80 flex flex-col justify-center items-center lg:items-end">
            <?php foreach ($properties as $index => $prop): ?>
                <div 
                    key="<?php echo $prop['id']; ?>" 
                    class="g-property-details details-<?php echo $index; ?> absolute inset-0 flex flex-col justify-center items-center lg:items-end <?php echo $index === 0 ? 'is-active' : 'opacity-0 pointer-events-none'; ?>"
                >
                    <h2 class="text-2xl sm:text-3xl lg:text-4xl font-medium text-gray-900 leading-tight"><?php echo htmlspecialchars($prop['title']); ?></h2>
                    <p class="text-sm sm:text-base text-gray-500 mt-2"><?php echo htmlspecialchars($prop['location']); ?></p>
                </div>
            <?php endforeach; ?>
        </div>
    </div>

    <!-- Center Content: Circular Image Viewer with SVG Animation -->
    <div class="g-about-ui-element relative w-[75vw] h-[75vw] max-w-[320px] max-h-[320px] md:w-[450px] md:h-[450px] md:max-w-none md:max-h-none lg:w-[550px] lg:h-[550px] my-8 lg:my-0 order-2">
        <svg viewBox="0 0 550 550" class="w-full h-full shadow-2xl rounded-full">
            <defs>
                <clipPath id="circle-clip"><circle cx="275" cy="275" r="275" /></clipPath>
            </defs>
            <g clipPath="url(#circle-clip)">
                <?php foreach ($properties as $index => $prop): ?>
                    <image 
                        href="<?php echo htmlspecialchars($prop['image']); ?>" 
                        class="property-image image-<?php echo $index; ?>"
                        width="100%" 
                        height="100%" 
                        preserveAspectRatio="xMidYMid slice"
                        style="opacity: <?php echo $index === 0 ? '1' : '0'; ?>;"
                    />
                <?php endforeach; ?>
            </g>
            <!-- This path is animated by GSAP to create the wipe effect -->
            <path
                id="wipePath"
                clipPath="url(#circle-clip)"
                fill="rgba(0, 0, 0, 0.35)" 
                stroke="rgba(255, 255, 255, 0.2)" 
                strokeWidth="1"
            />
        </svg>
    </div>

    <!-- Left Content: Property Index Numbers -->
    <div class="g-about-ui-element flex flex-row lg:flex-col items-center gap-8 z-20 order-3 lg:order-1 lg:absolute lg:left-12 lg:top-1/2 lg:-translate-y-1/2">
        <?php foreach ($properties as $index => $prop): ?>
            <div key="<?php echo $prop['id']; ?>" class="flex items-center gap-4">
                <span id="prop-id-<?php echo $index; ?>" class="text-3xl lg:text-5xl font-light transition-all duration-500 <?php echo $index === 0 ? 'text-gray-900 font-medium' : 'text-gray-400'; ?>">
                    <?php echo htmlspecialchars($prop['id']); ?>
                </span>
            </div>
        <?php endforeach; ?>
    </div>
    
    <!-- Bottom Left Content: "Best of the Month" -->
    <div class="g-about-ui-element text-sm text-gray-600 z-20 mt-12 lg:mt-0 order-4 lg:absolute lg:bottom-12 lg:left-12">
        Best of the Month
    </div>

    <!-- Bottom Right Content: "Book this Villa" Button -->
    <div class="g-about-ui-element z-20 mt-4 lg:mt-0 order-5 lg:absolute lg:bottom-12 lg:right-12">
        <button class="bg-gray-900 text-white text-sm font-medium px-6 py-3 rounded-full flex items-center gap-2 hover:bg-gray-700 hover:scale-105 hover:shadow-xl transition-all duration-300 ease-in-out shadow-lg">
            Book this Villa 
            <!-- Inlined SVG for the arrow icon -->
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-arrow-right"><line x1="5" y1="12" x2="19" y2="12"></line><polyline points="12 5 19 12 12 19"></polyline></svg>
        </button>
    </div>
</section>

<script>
document.addEventListener('DOMContentLoaded', () => {
    // --- GSAP Animation for AboutPreview Section ---

    // Ensure GSAP and ScrollTrigger are loaded
    if (typeof gsap === 'undefined' || typeof ScrollTrigger === 'undefined') {
        console.error('GSAP or ScrollTrigger is not loaded.');
        return;
    }

    const component = document.getElementById('about-preview-section');
    if (!component) return;

    // --- State and References ---
    let activeIndex = 0;
    const properties = <?php echo json_encode($properties); ?>;
    const wipePath = document.getElementById('wipePath');
    let timeline;

    // --- Function to Update UI based on activeIndex ---
    function updateActiveProperty(newIndex, oldIndex) {
        // Animate out the old property details
        if (oldIndex !== undefined) {
             gsap.to(`.details-${oldIndex} > *`, {
                opacity: 0,
                y: -20,
                duration: 0.4,
                stagger: 0.1,
                ease: 'power2.in',
            });
        }
       
        // Animate in the new property details
        gsap.fromTo(`.details-${newIndex} > *`, {
            opacity: 0,
            y: 20,
        }, {
            opacity: 1,
            y: 0,
            duration: 0.6,
            stagger: 0.1,
            ease: 'power2.out',
            delay: 0.4,
        });

        // Update the active classes for details containers
        document.querySelector(`.details-${oldIndex}`)?.classList.remove('is-active');
        document.querySelector(`.details-${newIndex}`)?.classList.add('is-active');

        // Fade out the old image and fade in the new one
        gsap.to(`.image-${oldIndex}`, {
            opacity: 0,
            duration: 0.8,
            ease: 'power2.inOut',
        });
        gsap.to(`.image-${newIndex}`, {
            opacity: 1,
            duration: 0.8,
            ease: 'power2.inOut',
        });

        // Update the style for the property ID numbers
        document.getElementById(`prop-id-${oldIndex}`)?.classList.remove('text-gray-900', 'font-medium');
        document.getElementById(`prop-id-${oldIndex}`)?.classList.add('text-gray-400');
        document.getElementById(`prop-id-${newIndex}`)?.classList.add('text-gray-900', 'font-medium');
        document.getElementById(`prop-id-${newIndex}`)?.classList.remove('text-gray-400');
    }

    // --- GSAP Timelines ---
    // This is the main timeline for the circular wipe animation
    timeline = gsap.timeline({
        paused: true,
        onComplete: () => {
            const oldIndex = activeIndex;
            const newIndex = (activeIndex + 1) % properties.length;
            activeIndex = newIndex;
            updateActiveProperty(newIndex, oldIndex);
            // Restart the timeline for a continuous loop
            timeline.restart();
        }
    });

    const wipeAnimation = { angle: 0 };
    timeline.to(wipeAnimation, {
        angle: 360,
        duration: 8,
        ease: 'none',
        onUpdate: () => {
            const angle = wipeAnimation.angle;
            const size = 550;
            const cx = size / 2;
            const cy = size / 2;
            const radius = size / 2;
            const startX = cx + radius * Math.cos(0);
            const startY = cy + radius * Math.sin(0);
            const endX = cx + radius * Math.cos(angle * Math.PI / 180);
            const endY = cy + radius * Math.sin(angle * Math.PI / 180);
            const largeArcFlag = angle <= 180 ? "0" : "1";
            const pathData = `M ${cx},${cy} L ${startX},${startY} A ${radius},${radius} 0 ${largeArcFlag} 1 ${endX},${endY} Z`;
            if (wipePath) {
                wipePath.setAttribute('d', pathData);
            }
        }
    });

    // This timeline controls the initial fade-in of UI elements on scroll
    const scrollTl = gsap.timeline({
        scrollTrigger: {
            trigger: component,
            start: 'top 50%',
            end: 'bottom top',
            toggleActions: 'play pause resume pause',
            onEnter: () => timeline.play(),
            onLeave: () => timeline.pause(),
            onEnterBack: () => timeline.resume(),
            onLeaveBack: () => timeline.pause(),
        }
    });

    scrollTl.from('.g-about-ui-element', {
        opacity: 0,
        y: 50,
        duration: 1,
        stagger: 0.1,
        ease: 'power3.out',
    });
});
</script>
