<?php
include '../config/db.php';

$owners = [];
$sql_owners = "SELECT `id`, `name`, `email`, `location`, `contact`, `status`, `registration_date` FROM `owners` WHERE 1 ORDER BY `name` ASC";
$result_owners = $conn->query($sql_owners);
if ($result_owners->num_rows > 0) {
    while($row = $result_owners->fetch_assoc()) {
        // Use owner ID as the key for easy lookup
        $owners[$row['id']] = $row;
    }
}

// 2. Fetch all properties and group them by owner_id for efficiency
$properties_by_owner = [];
$sql_properties = "SELECT `id`, `owner_id`, `property_type`, `rate`, `location`, `header`, `description`, `rules`, `image1`, `image2`, `image3`, `image4`, `upload_date`, `status` FROM `properties` WHERE 1 ORDER BY `upload_date` DESC";
$result_properties = $conn->query($sql_properties);
if ($result_properties->num_rows > 0) {
    while($row = $result_properties->fetch_assoc()) {
        $properties_by_owner[$row['owner_id']][] = $row;
    }
}

$conn->close();

// We will pass this data to JavaScript for building the modals dynamically
$app_data = json_encode([
    'owners' => $owners,
    'properties' => $properties_by_owner
]);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard | Owners & Properties</title>
    
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&family=Poppins:wght@300;400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">

    <style>
        /* --- General Styling & Variables --- */
        :root {
            --primary-color: #4F46E5;
            --primary-hover: #4338CA;
            --secondary-color: #6B7280;
            --bg-color: #F3F4F6;
            --card-bg: #FFFFFF;
            --text-color: #1F2937;
            --light-text: #6B7280;
            --border-color: #E5E7EB;
            --shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -2px rgba(0, 0, 0, 0.1);
            --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -4px rgba(0, 0, 0, 0.1);
            --success-color: #10B981;
            --danger-color: #EF4444;
        }

        body {
            font-family: 'Inter', sans-serif;
            background-color: var(--bg-color);
            color: var(--text-color);
            margin: 0;
            padding: 2rem;
        }

        /* --- Main Container & Header --- */
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 0 1.5rem;
        }

        header {
            text-align: center;
            margin-bottom: 3rem;
        }

        header h1 {
            font-family: 'Poppins', sans-serif;
            font-weight: 700;
            font-size: 2.75rem;
            color: var(--text-color);
        }

        header h1 .fa-shield-halved {
            color: var(--primary-color);
            margin-right: 12px;
            transform: translateY(2px);
        }

        /* --- Owner Grid --- */
        .owner-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
            gap: 1.5rem;
        }
        
        /* --- Owner Card --- */
        .owner-card {
            background: var(--card-bg);
            border-radius: 12px;
            box-shadow: var(--shadow);
            padding: 1.5rem;
            display: flex;
            flex-direction: column;
            justify-content: space-between;
            border: 1px solid var(--border-color);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .owner-card:hover {
            transform: translateY(-5px);
            box-shadow: var(--shadow-lg);
        }

        .owner-card-header {
            display: flex;
            align-items: flex-start;
            gap: 1rem;
            border-bottom: 1px solid var(--border-color);
            padding-bottom: 1rem;
            margin-bottom: 1rem;
        }

        .owner-avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: var(--primary-color);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            font-weight: 600;
            flex-shrink: 0;
        }

        .owner-card-header h3 {
            margin: 0;
            font-size: 1.25rem;
            font-weight: 600;
            line-height: 1.3;
        }

        .owner-card-header p {
            margin: 2px 0 0;
            color: var(--light-text);
            font-size: 0.9rem;
            word-break: break-all;
        }
        
        .owner-card-stats {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 1rem;
            margin-bottom: 1.5rem;
        }
        
        .stat-item {
            text-align: center;
        }
        
        .stat-item .count {
            font-size: 1.75rem;
            font-weight: 700;
            color: var(--primary-color);
        }
        
        .stat-item span {
            font-size: 0.8rem;
            color: var(--light-text);
            text-transform: uppercase;
        }
        
        .view-details-btn {
            width: 100%;
            background: var(--primary-color);
            color: white;
            border: none;
            padding: 0.85rem 1rem;
            border-radius: 8px;
            font-size: 1rem;
            font-weight: 500;
            cursor: pointer;
            transition: background-color 0.3s ease;
        }
        
        .view-details-btn:hover {
            background: var(--primary-hover);
        }
        
        .status {
            padding: 0.3rem 0.8rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
            color: #fff;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .status-active { background-color: var(--success-color); }
        .status-inactive { background-color: var(--danger-color); }

        /* --- MODAL STYLES --- */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.6);
            backdrop-filter: blur(5px);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 1000;
            opacity: 0;
            visibility: hidden;
            transition: opacity 0.3s ease, visibility 0.3s ease;
        }

        .modal-overlay.active {
            opacity: 1;
            visibility: visible;
        }

        .modal-container {
            background: var(--bg-color);
            width: 90%;
            max-width: 1100px;
            height: 90vh;
            border-radius: 16px;
            box-shadow: var(--shadow-lg);
            display: flex;
            flex-direction: column;
            transform: scale(0.95);
            transition: transform 0.3s ease;
        }
        
        .modal-overlay.active .modal-container {
            transform: scale(1);
        }

        .modal-header {
            padding: 1rem 1.5rem;
            border-bottom: 1px solid var(--border-color);
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-shrink: 0;
            background: var(--card-bg);
            border-radius: 16px 16px 0 0;
        }

        .modal-header h2 {
            font-size: 1.5rem;
            margin: 0;
            color: var(--text-color);
        }

        .close-modal-btn {
            background: none;
            border: none;
            font-size: 1.75rem;
            cursor: pointer;
            color: var(--secondary-color);
            transition: color 0.2s;
        }
        .close-modal-btn:hover {
            color: var(--text-color);
        }
        
        .modal-body {
            padding: 1.5rem;
            overflow-y: auto;
            flex-grow: 1;
        }
        
        .modal-tabs {
            display: flex;
            gap: 0.5rem;
            border-bottom: 1px solid var(--border-color);
            margin-bottom: 1.5rem;
        }
        
        .tab-btn {
            padding: 0.75rem 1.25rem;
            border: none;
            background: none;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            color: var(--light-text);
            border-bottom: 3px solid transparent;
            transition: color 0.3s, border-color 0.3s;
        }

        .tab-btn.active {
            color: var(--primary-color);
            border-bottom-color: var(--primary-color);
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
            animation: fadeIn 0.5s ease;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        /* Modal Content: Owner Details */
        .details-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1.25rem;
        }
        
        .detail-item {
            background: var(--card-bg);
            padding: 1rem;
            border-radius: 8px;
            display: flex;
            align-items: center;
            border: 1px solid var(--border-color);
        }
        
        .detail-item i {
            color: var(--primary-color);
            font-size: 1.2rem;
            margin-right: 1rem;
            width: 25px;
            text-align: center;
        }
        
        .detail-item .label {
            font-weight: 600;
            margin-right: 8px;
        }
        
        /* Modal Content: Properties */
        .property-card {
            background: var(--card-bg);
            border-radius: 12px;
            border: 1px solid var(--border-color);
            margin-bottom: 1.5rem;
            overflow: hidden;
            display: flex;
            flex-direction: column;
            box-shadow: var(--shadow);
        }
        
        .property-header {
            padding: 1rem 1.5rem;
            background-color: #F9FAFB;
            border-bottom: 1px solid var(--border-color);
        }
        
        .property-header h4 {
            margin: 0;
            font-size: 1.2rem;
            font-weight: 600;
        }
        
        .property-body {
            padding: 1.5rem;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1.5rem;
        }
        
        .property-details, .property-images {
            display: flex;
            flex-direction: column;
            gap: 1rem;
        }

        .property-detail-item { font-size: 0.95rem; }
        .property-detail-item i { margin-right: 10px; color: var(--secondary-color); width: 20px; }
        .property-detail-item strong { color: var(--text-color); font-weight: 600; }
        
        .property-images-container {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 10px;
        }

        .property-images-container img {
            width: 100%;
            height: 140px;
            object-fit: cover;
            border-radius: 8px;
            border: 2px solid var(--border-color);
        }
        
        .no-properties, .no-owners {
            text-align: center;
            padding: 3rem;
            color: var(--light-text);
            font-style: italic;
            font-size: 1.1rem;
            background: var(--card-bg);
            border-radius: 12px;
            border: 2px dashed var(--border-color);
            grid-column: 1 / -1; /* Make it span all columns if grid */
        }
    </style>
</head>
<body>

    <div class="container">
        <header>
            <h1><i class="fa-solid fa-shield-halved"></i>Admin Dashboard</h1>
        </header>

        <div class="owner-grid">
            <?php if (!empty($owners)): ?>
                <?php foreach ($owners as $owner): ?>
                    <?php
                        $owner_properties = $properties_by_owner[$owner['id']] ?? [];
                        $property_count = count($owner_properties);
                        $initial = strtoupper(substr($owner['name'], 0, 1));
                    ?>
                    <div class="owner-card">
                        <div>
                            <div class="owner-card-header">
                                <div class="owner-avatar"><?php echo htmlspecialchars($initial); ?></div>
                                <div>
                                    <h3><?php echo htmlspecialchars($owner['name']); ?></h3>
                                    <p><?php echo htmlspecialchars($owner['email']); ?></p>
                                </div>
                            </div>
                            <div class="owner-card-stats">
                                <div class="stat-item">
                                    <div class="count"><?php echo $property_count; ?></div>
                                    <span>Properties</span>
                                </div>
                                <div class="stat-item">
                                    <span class="status <?php echo ($owner['status'] == 1 ? 'status-active' : 'status-inactive'); ?>">
                                        <?php echo ($owner['status'] == 1 ? 'Active' : 'Inactive'); ?>
                                    </span>
                                    <span>Status</span>
                                </div>
                            </div>
                        </div>
                        <button class="view-details-btn" data-owner-id="<?php echo $owner['id']; ?>">
                            <i class="fa-solid fa-arrow-up-right-from-square"></i> View Details
                        </button>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <p class="no-owners"><i class="fa-solid fa-ghost"></i> No owners found in the database.</p>
            <?php endif; ?>
        </div>
    </div>

    <!-- MODAL STRUCTURE -->
    <div id="owner-modal" class="modal-overlay">
        <div class="modal-container">
            <header class="modal-header">
                <h2 id="modal-title">Owner Details</h2>
                <button class="close-modal-btn" id="close-modal">&times;</button>
            </header>
            <main class="modal-body" id="modal-body-content">
                <!-- Dynamic content will be injected here by JavaScript -->
            </main>
        </div>
    </div>


    <script>
        // Store PHP data in a JavaScript variable for easy access
        const appData = <?php echo $app_data; ?>;

        // DOM Elements
        const viewDetailsButtons = document.querySelectorAll('.view-details-btn');
        const modalOverlay = document.getElementById('owner-modal');
        const modalTitle = document.getElementById('modal-title');
        const modalBody = document.getElementById('modal-body-content');
        const closeModalBtn = document.getElementById('close-modal');

        // --- Functions ---

        /**
         * Formats a date string (e.g., "YYYY-MM-DD") into a more readable format (e.g., "August 9, 2025").
         * @param {string} dateString - The date string from the database.
         * @returns {string} The formatted date.
         */
        function formatDate(dateString) {
            if (!dateString) return 'N/A';
            const options = { year: 'numeric', month: 'long', day: 'numeric' };
            return new Date(dateString).toLocaleDateString(undefined, options);
        }
        
        /**
         * Creates the complete HTML content for the modal based on owner and property data.
         * @param {object} owner - The owner's data object.
         * @param {Array} properties - An array of the owner's properties.
         * @returns {string} The HTML string to be injected into the modal.
         */
        function createModalHtml(owner, properties) {
            const propertyCount = properties.length;
            
            // Generate Owner Details HTML
            const detailsHtml = `
                <div class="details-grid">
                    <div class="detail-item"><i class="fa-solid fa-id-card"></i><div><span class="label">Owner ID:</span> ${owner.id}</div></div>
                    <div class="detail-item"><i class="fa-solid fa-phone"></i><div><span class="label">Contact:</span> ${owner.contact || 'N/A'}</div></div>
                    <div class="detail-item"><i class="fa-solid fa-map-marker-alt"></i><div><span class="label">Location:</span> ${owner.location || 'N/A'}</div></div>
                    <div class="detail-item"><i class="fa-solid fa-calendar-alt"></i><div><span class="label">Registered:</span> ${formatDate(owner.registration_date)}</div></div>
                    <div class="detail-item"><i class="fa-solid fa-toggle-on"></i><div><span class="label">Status:</span> <span class="status ${owner.status == 1 ? 'status-active' : 'status-inactive'}">${owner.status == 1 ? 'Active' : 'Inactive'}</span></div></div>
                </div>
            `;
            
            // Generate Properties HTML
            let propertiesHtml = '';
            if (propertyCount > 0) {
                properties.forEach(prop => {
                    // **FIXED**: Use a more robust absolute path from the web root.
                    const imagePath = '/Hospitality/owner/uploads/';
                    let imagesHtml = '';
                    let hasImages = false;
                    for (let i = 1; i <= 4; i++) {
                        const imageField = `image${i}`;
                        if (prop[imageField]) {
                            hasImages = true;
                            imagesHtml += `<img src="${imagePath}${prop[imageField]}" alt="Property Image ${i}" onerror="this.style.display='none'">`;
                        }
                    }

                    propertiesHtml += `
                        <div class="property-card">
                            <div class="property-header">
                                <h4>${prop.header || 'Untitled Property'}</h4>
                            </div>
                            <div class="property-body">
                                <div class="property-details">
                                    <div class="property-detail-item"><i class="fa-solid fa-building"></i><strong>Type:</strong> ${prop.property_type || 'N/A'}</div>
                                    <div class="property-detail-item"><i class="fa-solid fa-tag"></i><strong>Rate:</strong> ${prop.rate || 'N/A'}</div>
                                    <div class="property-detail-item"><i class="fa-solid fa-location-dot"></i><strong>Location:</strong> ${prop.location || 'N/A'}</div>
                                    <div class="property-detail-item"><i class="fa-solid fa-file-alt"></i><strong>Description:</strong> ${prop.description ? prop.description.replace(/\n/g, '<br>') : 'N/A'}</div>
                                    <div class="property-detail-item"><i class="fa-solid fa-gavel"></i><strong>Rules:</strong> ${prop.rules ? prop.rules.replace(/\n/g, '<br>') : 'N/A'}</div>
                                    <div class="property-detail-item"><i class="fa-solid fa-calendar-check"></i><strong>Listed On:</strong> ${formatDate(prop.upload_date)}</div>
                                    <div class="property-detail-item">
                                        <i class="fa-solid fa-power-off"></i><strong>Status:</strong>
                                        <!-- **FIXED**: Changed labels to Active/Inactive for clarity -->
                                        <span class="status ${prop.status == 1 ? 'status-active' : 'status-inactive'}">
                                            ${prop.status == 1 ? 'Active' : 'Inactive'}
                                        </span>
                                    </div>
                                </div>
                                <div class="property-images">
                                    <strong><i class="fa-solid fa-images"></i> Images:</strong>
                                    <div class="property-images-container">
                                        ${hasImages ? imagesHtml : '<p>No images available.</p>'}
                                    </div>
                                </div>
                            </div>
                        </div>
                    `;
                });
            } else {
                propertiesHtml = '<p class="no-properties"><i class="fa-solid fa-ghost"></i> This owner has not listed any properties yet.</p>';
            }

            // Combine into final modal structure with tabs
            return `
                <div class="modal-tabs">
                    <button class="tab-btn active" data-tab="details"><i class="fa-solid fa-circle-info"></i> Owner Details</button>
                    <button class="tab-btn" data-tab="properties"><i class="fa-solid fa-hotel"></i> Properties (${propertyCount})</button>
                </div>
                <div id="details" class="tab-content active">
                    ${detailsHtml}
                </div>
                <div id="properties" class="tab-content">
                    ${propertiesHtml}
                </div>
            `;
        }
        
        /**
         * Opens the modal and populates it with data for the selected owner.
         * @param {string} ownerId - The ID of the owner to display.
         */
        function openModal(ownerId) {
            const owner = appData.owners[ownerId];
            const properties = appData.properties[ownerId] || [];
            
            if (!owner) {
                console.error("Owner not found for ID:", ownerId);
                alert("Could not find owner data. Please refresh the page.");
                return;
            }

            // Populate and show the modal
            modalTitle.textContent = `Details for ${owner.name}`;
            modalBody.innerHTML = createModalHtml(owner, properties);
            modalOverlay.classList.add('active');
            
            // Add event listeners for the new tab buttons inside the modal
            const tabButtons = modalBody.querySelectorAll('.tab-btn');
            tabButtons.forEach(button => {
                button.addEventListener('click', () => {
                    // Update button active state
                    modalBody.querySelector('.tab-btn.active').classList.remove('active');
                    button.classList.add('active');
                    
                    // Update content active state
                    modalBody.querySelector('.tab-content.active').classList.remove('active');
                    modalBody.querySelector(`#${button.dataset.tab}`).classList.add('active');
                });
            });
        }

        // --- Event Listeners ---

        // Open modal when a "View Details" button is clicked
        viewDetailsButtons.forEach(button => {
            button.addEventListener('click', () => {
                const ownerId = button.dataset.ownerId;
                openModal(ownerId);
            });
        });

        // Close modal using the 'X' button
        closeModalBtn.addEventListener('click', () => {
            modalOverlay.classList.remove('active');
        });

        // Close modal by clicking on the background overlay
        modalOverlay.addEventListener('click', (e) => {
            if (e.target === modalOverlay) {
                modalOverlay.classList.remove('active');
            }
        });

        // Close modal by pressing the Escape key
        document.addEventListener('keydown', (e) => {
            if (e.key === "Escape" && modalOverlay.classList.contains('active')) {
                modalOverlay.classList.remove('active');
            }
        });

    </script>
</body>
</html>
