<?php
// --- SESSION & SECURITY ---

// --- DATABASE CONNECTION ---
require_once '../config/db.php';

// --- FETCH ALL BOOKINGS ON INITIAL LOAD ---
$bookings = [];
// Updated SQL to also join the 'owners' table
$sql = "
    SELECT 
        b.id, b.booking_id, b.arrival_date, b.departure_date, b.adults, b.children, b.booking_date, b.status,
        u.name as user_name, u.email as user_email, u.contact as user_contact,
        p.header as property_header, p.location as property_location, p.owner_id,
        o.name as owner_name, o.email as owner_email, o.contact as owner_contact
    FROM bookings b
    JOIN users u ON b.user_id = u.id
    JOIN properties p ON b.property_id = p.id
    JOIN owners o ON p.owner_id = o.id
    ORDER BY b.booking_date DESC
";

$result = $conn->query($sql);
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $bookings[] = $row;
    }
}
$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - All Bookings</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Playfair+Display:wght@700&family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
    <style>
        body { font-family: 'Poppins', sans-serif; background-color: #f1f5f9; }
        .font-serif { font-family: 'Playfair Display', serif; }
        .gsap-reveal { opacity: 0; transform: translateY(30px); }
        .status-confirmed { background-color: #dcfce7; color: #166534; }
        .status-cancelled { background-color: #fee2e2; color: #991b1b; }
    </style>
</head>
<body class="text-gray-800">

    <main class="admin-main-container p-4 sm:p-6 lg:p-8">
        <div class="container mx-auto">
            <!-- HEADER SECTION -->
            <header class="admin-header mb-8">
                <div class="gsap-reveal flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
                    <div>
                        <h1 class="font-serif text-4xl md:text-5xl font-bold text-gray-900">All Bookings</h1>
                        <p class="text-gray-500 mt-1">Manage and view all property bookings.</p>
                    </div>
                    <div class="relative w-full sm:w-72">
                        <input type="text" id="booking-search-input" placeholder="Start typing to filter by ID..." class="w-full pl-10 pr-4 py-3 bg-white border border-gray-300 rounded-full focus:outline-none focus:ring-2 focus:ring-emerald-500 transition">
                        <i class="fas fa-search absolute left-4 top-1/2 -translate-y-1/2 text-gray-400"></i>
                    </div>
                </div>
            </header>

            <!-- BOOKINGS LIST SECTION -->
            <div class="bookings-list-container">
                <div id="bookings-list" class="space-y-6">
                    <?php if (!empty($bookings)): ?>
                        <?php foreach ($bookings as $index => $booking): ?>
                            <div class="booking-card bg-white rounded-2xl shadow-lg overflow-hidden transition-all duration-300 hover:shadow-xl" 
                                 data-aos="fade-up" 
                                 data-aos-delay="<?php echo $index * 50; ?>"
                                 data-booking-id="<?php echo htmlspecialchars($booking['booking_id']); ?>"
                                 data-user-name="<?php echo htmlspecialchars($booking['user_name']); ?>"
                                 data-user-email="<?php echo htmlspecialchars($booking['user_email']); ?>"
                                 data-user-contact="<?php echo htmlspecialchars($booking['user_contact']); ?>"
                                 data-owner-name="<?php echo htmlspecialchars($booking['owner_name']); ?>"
                                 data-owner-email="<?php echo htmlspecialchars($booking['owner_email']); ?>"
                                 data-owner-contact="<?php echo htmlspecialchars($booking['owner_contact']); ?>">
                                <div class="grid grid-cols-1 md:grid-cols-12 items-center">
                                    <div class="p-5 md:col-span-3 border-b md:border-b-0 md:border-r border-gray-200">
                                        <p class="text-sm text-gray-500">Booking ID</p>
                                        <p class="font-bold text-lg text-emerald-600"><?php echo htmlspecialchars($booking['booking_id']); ?></p>
                                        <p class="text-xs text-gray-400 mt-1"><?php echo date("d M Y, H:i", strtotime($booking['booking_date'])); ?></p>
                                    </div>
                                    <div class="p-5 md:col-span-4">
                                        <p class="font-bold text-xl text-gray-900 mb-1"><?php echo htmlspecialchars($booking['property_header']); ?></p>
                                        <div class="flex items-center gap-2 text-gray-500 text-sm">
                                            <i class="fas fa-map-marker-alt"></i><span><?php echo htmlspecialchars($booking['property_location']); ?></span>
                                        </div>
                                    </div>
                                    <div class="p-5 md:col-span-3">
                                        <div class="flex flex-wrap gap-2">
                                            <button class="view-user-btn text-sm bg-blue-100 text-blue-800 font-semibold px-3 py-1 rounded-full hover:bg-blue-200 transition">View User</button>
                                            <button class="view-owner-btn text-sm bg-purple-100 text-purple-800 font-semibold px-3 py-1 rounded-full hover:bg-purple-200 transition">View Owner</button>
                                        </div>
                                    </div>
                                    <div class="p-5 md:col-span-2 bg-gray-50/50 flex flex-col items-start gap-2">
                                        <p class="text-sm font-semibold text-gray-700">Check-in: <span class="font-normal text-gray-600"><?php echo date("d M Y", strtotime($booking['arrival_date'])); ?></span></p>
                                        <p class="text-sm font-semibold text-gray-700">Check-out: <span class="font-normal text-gray-600"><?php echo date("d M Y", strtotime($booking['departure_date'])); ?></span></p>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
                
                <div id="no-results-container" class="hidden text-center py-16 px-6 bg-white rounded-2xl shadow-md">
                    <div class="mx-auto flex items-center justify-center h-16 w-16 rounded-full bg-blue-100"><i class="fas fa-search-minus text-3xl text-blue-500"></i></div>
                    <h3 class="mt-5 text-2xl font-semibold text-gray-800">No Bookings Found</h3>
                    <p class="mt-2 text-md text-gray-500">We couldn't find any bookings matching your search.</p>
                </div>
            </div>
        </div>
    </main>

    <!-- DETAILS MODAL -->
    <div id="details-modal-container" class="hidden fixed inset-0 bg-black bg-opacity-60 z-50 flex items-center justify-center p-4">
        <div id="details-modal-box" class="bg-white rounded-2xl shadow-2xl w-full max-w-md overflow-hidden" style="transform: scale(0.7); opacity: 0;">
            <div id="details-modal-header" class="p-5 border-b border-gray-200 flex justify-between items-center">
                <h3 id="modal-title" class="font-serif text-2xl font-bold">Details</h3>
                <button id="close-modal-btn" class="text-gray-400 hover:text-gray-800 transition"><i class="fas fa-times text-xl"></i></button>
            </div>
            <div class="p-6 space-y-4">
                <div class="flex items-center gap-4">
                    <i id="modal-icon" class="text-2xl w-8 text-center"></i>
                    <p class="text-lg font-semibold" id="modal-name"></p>
                </div>
                <div class="flex items-center gap-4">
                    <i class="fas fa-envelope text-2xl w-8 text-center text-gray-400"></i>
                    <p class="text-gray-700" id="modal-email"></p>
                </div>
                <div class="flex items-center gap-4">
                    <i class="fas fa-phone text-2xl w-8 text-center text-gray-400"></i>
                    <p class="text-gray-700" id="modal-contact"></p>
                </div>
            </div>
        </div>
    </div>

    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/gsap/3.12.5/gsap.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', () => {
            AOS.init({ once: true, duration: 600, easing: 'ease-in-out-cubic' });

            const tl = gsap.timeline({ defaults: { ease: 'power3.out' } });
            tl.staggerFromTo('.gsap-reveal', 1, { opacity: 0, y: 30 }, { opacity: 1, y: 0, stagger: 0.2 });

            const searchInput = document.getElementById('booking-search-input');
            const bookingCards = document.querySelectorAll('.booking-card');
            const noResultsContainer = document.getElementById('no-results-container');

            searchInput.addEventListener('input', function() {
                const searchTerm = this.value.toUpperCase().trim();
                let visibleCount = 0;
                const filterTl = gsap.timeline();
                bookingCards.forEach(card => {
                    const bookingId = card.dataset.bookingId.toUpperCase();
                    if (bookingId.includes(searchTerm)) {
                        filterTl.to(card, { opacity: 1, height: 'auto', display: 'block', marginTop: '1.5rem', duration: 0.3 }, "<");
                        visibleCount++;
                    } else {
                        filterTl.to(card, { opacity: 0, height: 0, display: 'none', marginTop: 0, duration: 0.3 }, "<");
                    }
                });
                gsap.to(noResultsContainer, { display: visibleCount === 0 ? 'block' : 'none', opacity: visibleCount === 0 ? 1 : 0, duration: 0.4 });
            });

            // --- MODAL LOGIC ---
            const modalContainer = document.getElementById('details-modal-container');
            const modalBox = document.getElementById('details-modal-box');
            const closeModalBtn = document.getElementById('close-modal-btn');
            const modalTitle = document.getElementById('modal-title');
            const modalIcon = document.getElementById('modal-icon');
            const modalName = document.getElementById('modal-name');
            const modalEmail = document.getElementById('modal-email');
            const modalContact = document.getElementById('modal-contact');

            function openModal() {
                modalContainer.classList.remove('hidden');
                gsap.to(modalBox, { scale: 1, opacity: 1, duration: 0.4, ease: 'power3.out' });
            }

            function closeModal() {
                gsap.to(modalBox, { scale: 0.7, opacity: 0, duration: 0.3, ease: 'power3.in', onComplete: () => {
                    modalContainer.classList.add('hidden');
                }});
            }

            document.querySelectorAll('.view-user-btn').forEach(button => {
                button.addEventListener('click', function() {
                    const card = this.closest('.booking-card');
                    modalTitle.textContent = 'User Details';
                    modalIcon.className = 'fas fa-user text-2xl w-8 text-center text-blue-500';
                    modalName.textContent = card.dataset.userName;
                    modalEmail.textContent = card.dataset.userEmail;
                    modalContact.textContent = card.dataset.userContact;
                    openModal();
                });
            });

            document.querySelectorAll('.view-owner-btn').forEach(button => {
                button.addEventListener('click', function() {
                    const card = this.closest('.booking-card');
                    modalTitle.textContent = 'Owner Details';
                    modalIcon.className = 'fas fa-store-alt text-2xl w-8 text-center text-purple-500';
                    modalName.textContent = card.dataset.ownerName;
                    modalEmail.textContent = card.dataset.ownerEmail;
                    modalContact.textContent = card.dataset.ownerContact;
                    openModal();
                });
            });

            closeModalBtn.addEventListener('click', closeModal);
            modalContainer.addEventListener('click', (e) => {
                if (e.target === modalContainer) {
                    closeModal();
                }
            });
        });
    </script>

</body>
</html>
