<?php // FILE: owner_req.php (Updated with Login Protection) ?>
<?php
// Start the session at the very beginning
session_start();

// --- Require Admin Login ---
// This assumes your admin login sets $_SESSION['admin_id'] when successful.
if (!isset($_SESSION['admin_id'])) {
    header("Location: admin_login.php");
    exit();
}

// Include the database connection file
require_once "../config/db.php";

// --- Handle Status Update Action ---
if (isset($_GET['action']) && isset($_GET['id'])) {
    $action = $_GET['action'];
    $id = intval($_GET['id']); // Prevent SQL injection

    $new_status = '';
    if ($action == 'approve') {
        $new_status = 'approved';
    } elseif ($action == 'reject') {
        $new_status = 'rejected';
    }

    if (!empty($new_status)) {
        $sql = "UPDATE owners SET status = ? WHERE id = ?";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, "si", $new_status, $id);
            if (mysqli_stmt_execute($stmt)) {
                $_SESSION['flash_message'] = "Request Accepted! The owner's status has been updated to '" . htmlspecialchars($new_status) . "'.";
                $_SESSION['flash_message_class'] = 'bg-green-500/20 text-green-300';
            } else {
                $_SESSION['flash_message'] = "Error: Could not update the owner's status.";
                $_SESSION['flash_message_class'] = 'bg-red-500/20 text-red-300';
            }
            mysqli_stmt_close($stmt);
        }
        header("location: owner_req.php");
        exit();
    }
}

// --- Fetch All Owners for Display ---
$owners = [];
$sql_fetch = "SELECT id, name, email, location, contact, status FROM owners ORDER BY registration_date DESC";
$result = mysqli_query($link, $sql_fetch);
if ($result) {
    $owners = mysqli_fetch_all($result, MYSQLI_ASSOC);
    mysqli_free_result($result);
}

// Close DB connection
mysqli_close($link);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/gsap/3.9.1/gsap.min.js"></script>
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script src="https://unpkg.com/lucide@latest"></script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap');
        body { font-family: 'Inter', sans-serif; background-color: #111827; }
        .glassmorphism { background: rgba(255, 255, 255, 0.05); backdrop-filter: blur(12px); -webkit-backdrop-filter: blur(12px); border: 1px solid rgba(255, 255, 255, 0.1); }
        .status-pill { padding: 0.25rem 0.75rem; border-radius: 9999px; font-size: 0.8rem; font-weight: 600; text-transform: capitalize;}
        .status-pending { background-color: rgba(251, 191, 36, 0.2); color: #FBBF24; }
        .status-approved { background-color: rgba(52, 211, 153, 0.2); color: #34D399; }
        .status-rejected { background-color: rgba(248, 113, 113, 0.2); color: #F87171; }
    </style>
</head>
<body class="text-white">
    <div class="container mx-auto p-4 md:p-8">
        <div data-aos="fade-down">
            <h1 class="text-4xl font-bold text-center mb-2 text-cyan-300">Admin Dashboard</h1>
            <p class="text-center text-gray-400 mb-8">Manage Owner Registrations</p>
        </div>

        <?php if (isset($_SESSION['flash_message'])): ?>
            <div id="flash-message" class="p-4 mb-6 rounded-lg font-semibold <?php echo $_SESSION['flash_message_class']; ?>" role="alert">
                <?php 
                    echo $_SESSION['flash_message']; 
                    unset($_SESSION['flash_message']);
                    unset($_SESSION['flash_message_class']);
                ?>
            </div>
        <?php endif; ?>
        
        <div class="glassmorphism rounded-2xl p-4 md:p-6 shadow-2xl overflow-x-auto" data-aos="fade-up">
            <table class="w-full text-left min-w-[800px]">
                <thead>
                    <tr class="border-b border-gray-700">
                        <th class="p-4"><i data-lucide="user" class="inline-block mr-2"></i>Name</th>
                        <th class="p-4"><i data-lucide="mail" class="inline-block mr-2"></i>Email</th>
                        <th class="p-4"><i data-lucide="map-pin" class="inline-block mr-2"></i>Location</th>
                        <th class="p-4"><i data-lucide="phone" class="inline-block mr-2"></i>Contact</th>
                        <th class="p-4 text-center"><i data-lucide="activity" class="inline-block mr-2"></i>Status</th>
                        <th class="p-4 text-center"><i data-lucide="settings" class="inline-block mr-2"></i>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($owners)): ?>
                        <tr>
                            <td colspan="6" class="text-center p-8 text-gray-400">No registrations found.</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($owners as $owner): ?>
                            <tr class="border-b border-gray-800 hover:bg-gray-800/50 transition-colors duration-300 owner-row">
                                <td class="p-4"><?php echo htmlspecialchars($owner['name']); ?></td>
                                <td class="p-4"><?php echo htmlspecialchars($owner['email']); ?></td>
                                <td class="p-4"><?php echo htmlspecialchars($owner['location']); ?></td>
                                <td class="p-4"><?php echo htmlspecialchars($owner['contact']); ?></td>
                                <td class="p-4 text-center">
                                    <span class="status-pill status-<?php echo htmlspecialchars($owner['status']); ?>">
                                        <?php echo htmlspecialchars($owner['status']); ?>
                                    </span>
                                </td>
                                <td class="p-4 text-center space-x-2">
                                    <a href="owner_req.php?action=approve&id=<?php echo $owner['id']; ?>" class="inline-block bg-green-500 hover:bg-green-600 text-white p-2 rounded-full transition-transform transform hover:scale-110" title="Approve">
                                        <i data-lucide="check" class="w-4 h-4"></i>
                                    </a>
                                    <a href="owner_req.php?action=reject&id=<?php echo $owner['id']; ?>" class="inline-block bg-red-500 hover:bg-red-600 text-white p-2 rounded-full transition-transform transform hover:scale-110" title="Reject">
                                        <i data-lucide="x" class="w-4 h-4"></i>
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
    <script>
        AOS.init({ duration: 800, once: true });
        lucide.createIcons();

        gsap.from(".owner-row", {
            duration: 0.5,
            opacity: 0,
            x: -50,
            stagger: 0.1,
            ease: "power2.out",
            delay: 0.3
        });

        const flashMessage = document.getElementById('flash-message');
        if (flashMessage) {
            gsap.fromTo(flashMessage, {opacity: 0, y: -20}, {opacity: 1, y: 0, duration: 0.5, ease: 'power2.out'});
            setTimeout(() => {
                gsap.to(flashMessage, {opacity: 0, duration: 0.5, onComplete: () => flashMessage.remove()});
            }, 5000);
        }
    </script>
</body>
</html>
