<?php
// --- DATABASE CONNECTION ---

include "../config/db.php"; // Include your database connection file
// --- CONFIGURATION ---
$targetDir = "../owner/uploads/"; // The directory where images will be stored
$uploadSuccess = null; // To store success or error messages

// Create uploads folder if it doesn't exist
if (!file_exists($targetDir)) {
    mkdir($targetDir, 0777, true);
}

// --- DELETE IMAGE LOGIC ---
if (isset($_GET['delete_id'])) {
    $imageId = intval($_GET['delete_id']);

    // 1. Get the image path from the DB to delete the file
    $stmt = $link->prepare("SELECT image_path FROM hero_poster WHERE id = ?");
    $stmt->bind_param("i", $imageId);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $filePath = $row['image_path'];

        // 2. Delete the file from the server
        if (file_exists($filePath)) {
            unlink($filePath);
        }

        // 3. Delete the record from the database
        $deleteStmt = $link->prepare("DELETE FROM hero_poster WHERE id = ?");
        $deleteStmt->bind_param("i", $imageId);
        $deleteStmt->execute();
        $deleteStmt->close();
    }
    $stmt->close();

    // 4. Redirect to clean the URL and prevent accidental re-deletion
    header("Location: " . strtok($_SERVER["REQUEST_URI"], '?'));
    exit();
}


// --- UPLOAD IMAGE LOGIC ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_FILES["hero_image"]) && $_FILES["hero_image"]["error"] == 0) {
    $fileName = basename($_FILES["hero_image"]["name"]);
    // Sanitize filename to prevent directory traversal issues
    $safeFileName = preg_replace("/[^a-zA-Z0-9\.\-\_]/", "", $fileName);
    $targetFilePath = $targetDir . time() . '-' . $safeFileName; // Prepend timestamp to avoid name conflicts

    // Check if file is a valid image
    $imageFileType = strtolower(pathinfo($targetFilePath, PATHINFO_EXTENSION));
    $allowTypes = array('jpg', 'png', 'jpeg', 'gif', 'webp');
    if (in_array($imageFileType, $allowTypes)) {
        // Move the uploaded file to the target directory
        if (move_uploaded_file($_FILES["hero_image"]["tmp_name"], $targetFilePath)) {
            // Insert image file path into the database
            $stmt = $link->prepare("INSERT INTO hero_poster (image_path) VALUES (?)");
            $stmt->bind_param("s", $targetFilePath);
            if ($stmt->execute()) {
                $uploadSuccess = "Image uploaded successfully!";
            } else {
                $uploadSuccess = "Database insertion failed.";
            }
            $stmt->close();
        } else {
            $uploadSuccess = "Sorry, there was an error uploading your file.";
        }
    } else {
        $uploadSuccess = 'Sorry, only JPG, JPEG, PNG, GIF, & WEBP files are allowed.';
    }
} elseif ($_SERVER["REQUEST_METHOD"] == "POST") {
    $uploadSuccess = "Error: " . $_FILES["hero_image"]["error"];
}

// --- FETCH IMAGES FOR DISPLAY ---
// Fetch all images, ordering by the newest first
$images = [];
$result = $link->query("SELECT id, image_path FROM hero_poster ORDER BY id DESC");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $images[] = $row;
    }
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Hero Image Management</title>
    
    <!-- Tailwind CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    
    <!-- AOS Animation Library -->
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
    
    <!-- Font Awesome Icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    
    <!-- Google Fonts -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;700&display=swap" rel="stylesheet">

    <style>
        /* Custom Styles */
        body {
            font-family: 'Inter', sans-serif;
            background-color: #111827; /* bg-gray-900 */
        }
        /* Custom file input styling */
        .file-input-wrapper {
            position: relative;
            overflow: hidden;
            display: inline-block;
        }
        .file-input-wrapper input[type=file] {
            font-size: 100px;
            position: absolute;
            left: 0;
            top: 0;
            opacity: 0;
            cursor: pointer;
        }
        .file-input-button {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            padding: 0.75rem 1.5rem;
            border: 2px solid #374151; /* border-gray-700 */
            border-radius: 0.5rem;
            color: white;
            background-color: #1f2937; /* bg-gray-800 */
            cursor: pointer;
            transition: all 0.3s ease;
        }
        .file-input-wrapper:hover .file-input-button {
            background-color: #374151; /* bg-gray-700 */
            border-color: #4f46e5; /* border-indigo-600 */
        }
        .file-name {
            margin-left: 1rem;
            color: #9ca3af; /* text-gray-400 */
        }
        /* Confirmation Modal */
        .modal {
            transition: opacity 0.25s ease;
        }
    </style>
</head>
<body class="text-gray-200">

    <div class="container mx-auto px-4 sm:px-6 lg:px-8 py-12">

        <!-- Header -->
        <div class="text-center mb-12" data-aos="fade-down">
            <h1 class="text-4xl sm:text-5xl font-bold text-white">Hero Image Manager</h1>
            <p class="mt-4 text-lg text-gray-400">Upload, view, and manage your hero section images.</p>
        </div>

        <!-- Upload Form -->
        <div class="max-w-2xl mx-auto bg-gray-800 rounded-xl shadow-lg p-8 mb-12" data-aos="fade-up">
            <form action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="post" enctype="multipart/form-data" id="uploadForm">
                <div class="flex flex-col items-center justify-center w-full">
                    <div class="file-input-wrapper">
                        <input type="file" name="hero_image" id="hero_image" required>
                        <div class="file-input-button">
                            <i class="fas fa-cloud-upload-alt mr-3"></i>
                            <span>Choose an image...</span>
                        </div>
                    </div>
                    <span id="fileName" class="file-name mt-4">No file selected</span>
                </div>
                <div class="mt-6 text-center">
                    <button type="submit" class="w-full sm:w-auto bg-indigo-600 hover:bg-indigo-700 text-white font-bold py-3 px-8 rounded-lg shadow-md transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 focus:ring-offset-gray-800">
                        <i class="fas fa-upload mr-2"></i> Upload Hero Image
                    </button>
                </div>
            </form>
            <?php if ($uploadSuccess): ?>
                <div class="mt-4 text-center p-3 rounded-lg <?php echo strpos($uploadSuccess, 'successfully') !== false ? 'bg-green-500/20 text-green-300' : 'bg-red-500/20 text-red-300'; ?>">
                    <?php echo htmlspecialchars($uploadSuccess); ?>
                </div>
            <?php endif; ?>
        </div>
        
        <hr class="border-gray-700 my-12" data-aos="fade-up">

        <!-- Image Gallery -->
        <div class="pb-4" data-aos="fade-up">
             <h2 class="text-3xl font-bold text-white mb-8 text-center">Uploaded Images</h2>
             <?php if (!empty($images)): ?>
                <div class="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-6">
                    <?php foreach ($images as $index => $image): ?>
                        <div class="relative group bg-gray-800 rounded-lg overflow-hidden shadow-lg" data-aos="zoom-in" data-aos-delay="<?php echo ($index % 4) * 100; ?>">
                            <!-- Active Badge for the newest image -->
                            <?php if ($index === 0): ?>
                                <span class="absolute top-3 right-3 bg-indigo-600 text-white text-xs font-bold px-3 py-1 rounded-full z-10 shadow-lg">Active</span>
                            <?php endif; ?>
                            
                            <img src="<?php echo htmlspecialchars($image['image_path']); ?>" alt="Uploaded Image" class="w-full h-56 object-cover group-hover:scale-110 transition-transform duration-500 ease-in-out">
                            
                            <!-- Delete Button Overlay -->
                            <div class="absolute inset-0 bg-black bg-opacity-0 group-hover:bg-opacity-60 flex items-center justify-center transition-all duration-300">
                                <a href="?delete_id=<?php echo $image['id']; ?>" 
                                   class="text-white text-4xl opacity-0 group-hover:opacity-100 transform scale-75 group-hover:scale-100 transition-all duration-300"
                                   onclick="return confirm('Are you sure you want to permanently delete this image?');"
                                   title="Delete Image">
                                    <i class="fas fa-trash-alt"></i>
                                </a>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
             <?php else: ?>
                <div class="text-center py-16 px-6 bg-gray-800 rounded-lg" data-aos="fade-up">
                    <i class="fas fa-images text-5xl text-gray-500 mb-4"></i>
                    <h3 class="text-2xl font-semibold text-white">No Images Found</h3>
                    <p class="text-gray-400 mt-2">Upload your first image to see it appear here.</p>
                </div>
             <?php endif; ?>
        </div>

    </div>

    <!-- AOS JS -->
    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script>
        // Initialize AOS
        AOS.init({
            duration: 800,
            once: true,
        });

        // Script to display selected file name
        const fileInput = document.getElementById('hero_image');
        const fileNameSpan = document.getElementById('fileName');
        if(fileInput) {
            fileInput.addEventListener('change', function() {
                if (this.files.length > 0) {
                    fileNameSpan.textContent = this.files[0].name;
                } else {
                    fileNameSpan.textContent = 'No file selected';
                }
            });
        }
    </script>
</body>
</html>
