<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>What We Offer - Enhanced Design</title>
    
    <!-- Tailwind CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    
    <!-- GSAP (Animation Library) -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/gsap/3.12.2/gsap.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/gsap/3.12.2/ScrollTrigger.min.js"></script>
    
    <!-- Google Fonts -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600&display=swap" rel="stylesheet">

    <style>
        /* Using Inter font as the base */
        body {
            font-family: 'Inter', sans-serif;
        }
        /* Custom class for the frosted glass effect */
        .frosted-glass {
            background-color: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(12px);
            -webkit-backdrop-filter: blur(12px); /* For Safari */
            border: 1px solid rgba(255, 255, 255, 0.15);
        }
    </style>
</head>
<body class="bg-black">

<!-- 
  SECTION: What We Offer
  This section displays a grid of services.
  - Fully responsive layout that adapts from a single column to a large grid.
  - Scroll-triggered animations for the section entrance.
  - A stylish "frosted glass" hover effect on the service items.
-->
<section id="what-we-offer-section" class="relative w-full min-h-screen h-auto flex items-center py-16 lg:py-0 text-white">
    
    <!-- Background Image and Dark Overlay -->
    <div class="absolute inset-0 z-0">
        <!-- Using a high-quality image for a premium feel -->
        <img src="https://images.unsplash.com/photo-1613490493576-7fde63acd811?q=80&w=2671&auto=format&fit=crop" 
             alt="Luxury Villa by a pool at dusk" 
             class="w-full h-full object-cover"
             onerror="this.onerror=null;this.src='https://placehold.co/1920x1080/000000/ffffff?text=Luxury+Villa';">
        <div class="absolute inset-0 bg-black/60"></div>
    </div>

    <!-- Main Content Container -->
    <div class="relative z-10 w-full max-w-7xl mx-auto flex flex-col lg:flex-row items-center">
        
        <!-- Left Side: Title and Description -->
        <div class="g-offer-title w-full lg:w-1/3 p-8 lg:pr-16 flex flex-col justify-center text-center lg:text-left">
            <h2 class="text-4xl md:text-5xl font-medium leading-tight">Our services approach is designed to create a seamless and unforgettable experience.</h2>
            <p class="text-white/70 mt-6 max-w-md mx-auto lg:mx-0">From the moment they step into our world of exclusivity, we cater to every desire with unmatched precision and care.</p>
        </div>

        <!-- Right Side: Services Grid -->
        <!-- This grid is now fully responsive. It changes from 2 columns on mobile to 3 on desktop. -->
        <div class="g-offer-grid grid grid-cols-2 lg:grid-cols-3 w-full lg:w-2/3 h-auto lg:h-[75vh] mt-12 lg:mt-0 border-t border-l border-white/10">
            
            <!-- Service Cell: Event Planning -->
            <div class="g-grid-cell relative border-r border-b border-white/10 p-6 md:p-8 flex items-end group cursor-pointer">
                <div class="cell-overlay frosted-glass absolute inset-0 origin-center" style="transform: scale(0);"></div>
                <span class="cell-text relative z-10 text-white text-lg font-medium">Event Planning</span>
            </div>

            <!-- Service Cell: Dedicated Concierge -->
            <div class="g-grid-cell relative border-r border-b border-white/10 p-6 md:p-8 flex items-end group cursor-pointer">
                <div class="cell-overlay frosted-glass absolute inset-0 origin-center" style="transform: scale(0);"></div>
                <span class="cell-text relative z-10 text-white text-lg font-medium">Dedicated Concierge</span>
            </div>
            
            <!-- Service Cell: Personal Chef -->
            <div class="g-grid-cell relative border-r border-b border-white/10 p-6 md:p-8 flex items-end group cursor-pointer">
                <div class="cell-overlay frosted-glass absolute inset-0 origin-center" style="transform: scale(0);"></div>
                <span class="cell-text relative z-10 text-white text-lg font-medium">Personal Chef</span>
            </div>

            <!-- Service Cell: Spa and Wellness (Center cell with special content) -->
            <div class="g-grid-cell relative border-r border-b border-white/10 p-6 md:p-8 flex items-end justify-center group cursor-pointer">
                <div class="cell-overlay frosted-glass absolute inset-0 origin-center" style="transform: scale(0);"></div>
                <div class="relative z-10 text-center">
                    <span class="cell-text text-white text-lg font-medium">Spa and Wellness</span>
                    <!-- Detail indicator that appears on hover -->
                    <div class="cell-details absolute top-full left-1/2 -translate-x-1/2 opacity-0 mt-4 w-max">
                        <div class="w-16 h-16 mx-auto rounded-full border border-gray-400 flex items-center justify-center">
                            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round" class="text-gray-300"><line x1="12" y1="5" x2="12" y2="19"></line><polyline points="19 12 12 19 5 12"></polyline></svg>
                        </div>
                        <p class="text-center text-gray-300 text-sm mt-2">View Details</p>
                    </div>
                </div>
            </div>

            <!-- Service Cell: Private Security -->
            <div class="g-grid-cell relative border-r border-b border-white/10 p-6 md:p-8 flex items-end group cursor-pointer">
                <div class="cell-overlay frosted-glass absolute inset-0 origin-center" style="transform: scale(0);"></div>
                <span class="cell-text relative z-10 text-white text-lg font-medium">Private Security</span>
            </div>

            <!-- Service Cell: Yacht Charters -->
            <div class="g-grid-cell relative border-r border-b border-white/10 p-6 md:p-8 flex items-end group cursor-pointer">
                <div class="cell-overlay frosted-glass absolute inset-0 origin-center" style="transform: scale(0);"></div>
                <span class="cell-text relative z-10 text-white text-lg font-medium">Yacht Charters</span>
            </div>

            <!-- Service Cell: Luxury Transport -->
            <div class="g-grid-cell relative border-r border-b border-white/10 p-6 md:p-8 flex items-end group cursor-pointer">
                <div class="cell-overlay frosted-glass absolute inset-0 origin-center" style="transform: scale(0);"></div>
                <span class="cell-text relative z-10 text-white text-lg font-medium">Luxury Transport</span>
            </div>

            <!-- These two are empty cells for layout, only visible on large screens to complete the 3x3 grid -->
            <div class="hidden lg:block border-r border-b border-white/10"></div>
            <div class="hidden lg:block border-r border-b border-white/10"></div>

        </div>
    </div>
</section>

<script>
document.addEventListener('DOMContentLoaded', () => {
    // Check if GSAP and ScrollTrigger are loaded
    if (typeof gsap === 'undefined' || typeof ScrollTrigger === 'undefined') {
        console.error('GSAP or ScrollTrigger is not loaded.');
        return;
    }

    const component = document.getElementById('what-we-offer-section');
    if (!component) return;

    // --- 1. Main Scroll-Triggered Animation ---
    // Animates the title and grid into view when the section is scrolled to.
    const mainTl = gsap.timeline({
        scrollTrigger: {
            trigger: component,
            start: 'top 60%', // Start animation when 60% of the section is visible
            toggleActions: 'play none none reverse',
        }
    });

    // Animate the title and description
    mainTl.from('.g-offer-title', {
        opacity: 0,
        y: 50,
        duration: 1,
        ease: 'power3.out'
    })
    // Animate the grid cells with a stagger effect
    .from('.g-grid-cell', {
        scale: 0.95,
        opacity: 0,
        stagger: 0.1, // Stagger animation for each cell
        duration: 0.8,
        ease: 'power3.out'
    }, 0.3); // Start 0.3s after the title animation begins

    // --- 2. Grid Cell Hover Animation ---
    // Sets up the stylish "frosted glass" hover effect for each service cell.
    const gridCells = document.querySelectorAll('.g-grid-cell');

    gridCells.forEach(cell => {
        // Find the interactive elements within each cell
        const overlay = cell.querySelector('.cell-overlay');
        const text = cell.querySelector('.cell-text');
        const details = cell.querySelector('.cell-details');

        // If a cell has no text, it's a spacer. Do nothing.
        if (!text) return;

        // Create a unique, paused timeline for each cell's hover animation
        const hoverTl = gsap.timeline({ paused: true });

        // The main animation: scale up the frosted glass overlay.
        // The text color no longer needs to change.
        hoverTl.to(overlay, {
            scale: 1,
            duration: 0.6,
            ease: 'power3.inOut'
        })
        // Also, subtly scale up the text to make it feel more interactive.
        .to(text, {
            scale: 1.05,
            duration: 0.6,
            ease: 'power3.inOut'
        }, 0); // The '0' at the end makes this animation start at the same time as the one above.

        // If the cell has special details (like the center cell), animate them.
        if (details) {
            hoverTl.to(details, {
                opacity: 1,
                y: -10, // Move it up slightly
                duration: 0.5,
                ease: 'power2.out'
            }, 0.2); // Starts 0.2s into the hover animation
        }

        // Use mouseenter and mouseleave events to play and reverse the timeline.
        cell.addEventListener('mouseenter', () => hoverTl.play());
        cell.addEventListener('mouseleave', () => hoverTl.reverse());
    });
});
</script>

</body>
</html>
