<?php
// FILE: /actions/booking_handler.php

// --- SESSION & DATABASE SETUP ---
require_once 'config/db.php';

// --- FORM DATA PROCESSING ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // --- 1. GATHER AND SANITIZE DATA ---
    $property_id = isset($_POST['property_id']) ? intval($_POST['property_id']) : 0;
    $user_id = $_SESSION['user_id']; // Get logged-in user's ID
    $arrival_date_str = $_POST['arrive_date'] ?? '';
    $departure_date_str = $_POST['depart_date'] ?? '';
    $adults = isset($_POST['adults']) ? intval($_POST['adults']) : 0;
    $children = isset($_POST['child']) ? intval($_POST['child']) : 0;

    // --- 2. VALIDATE INPUT ---
    if ($property_id === 0 || empty($arrival_date_str) || empty($departure_date_str) || $adults === 0) {
        // Redirect back with an error message
        header("Location: ../property_details.php?id={$property_id}&error=missing_fields");
        exit;
    }

    // Convert dates and validate them
    $arrival_date = new DateTime($arrival_date_str);
    $departure_date = new DateTime($departure_date_str);
    $today = new DateTime();

    if ($arrival_date < $today || $departure_date <= $arrival_date) {
        // Redirect back with a date error
        header("Location: ../property_details.php?id={$property_id}&error=invalid_dates");
        exit;
    }

    // --- 3. FETCH PROPERTY DETAILS & CALCULATE TOTAL RATE ---
    $stmt = $link->prepare("SELECT owner_id, header, rate FROM properties WHERE id = ?");
    $stmt->bind_param("i", $property_id);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        // Redirect back with a property not found error
        header("Location: ../available_properties.php?error=property_not_found");
        exit;
    }

    $property = $result->fetch_assoc();
    $owner_id = $property['owner_id'];
    $property_header = $property['header'];
    $rate_per_night = $property['rate'];

    // Calculate the number of nights
    $interval = $arrival_date->diff($departure_date);
    $num_nights = $interval->days;

    // Calculate total rate
    $total_rate = $num_nights * $rate_per_night;
    $stmt->close();

    // --- 4. INSERT BOOKING INTO DATABASE ---
    $sql = "INSERT INTO bookings (property_id, owner_id, user_id, property_header, arrival_date, departure_date, adults, children, total_rate) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    if ($stmt_insert = $link->prepare($sql)) {
        $stmt_insert->bind_param(
            "iiisssiid",
            $property_id,
            $owner_id,
            $user_id,
            $property_header,
            $arrival_date_str,
            $departure_date_str,
            $adults,
            $children,
            $total_rate
        );

        if ($stmt_insert->execute()) {
            // Success! Redirect to a confirmation page or user's dashboard.
            // We'll pass the new booking ID to the success page.
            $new_booking_id = $stmt_insert->insert_id;
            header("Location: ../booking_success.php?booking_id={$new_booking_id}");
            exit;
        } else {
            // Database insertion error
            header("Location: ../property_details.php?id={$property_id}&error=db_error");
            exit;
        }
        $stmt_insert->close();
    }
} else {
    // If not a POST request, redirect away
    header('Location: ../available_properties.php');
    exit;
}

$link->close();
?>
