<?php
// --- SESSION & SECURITY ---
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// If the user is not logged in, redirect them to the login page
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
    header("location: login.php");
    exit;
}

// --- DATABASE CONNECTION ---
require_once 'config/db.php';

// --- HELPER FUNCTIONS ---
/**
 * Calculates the number of nights between two dates.
 * @param string $start_date
 * @param string $end_date
 * @return int The total number of nights.
 */
function calculate_nights($start_date, $end_date) {
    $start = new DateTime($start_date);
    $end = new DateTime($end_date);
    $interval = $start->diff($end);
    return $interval->days > 0 ? $interval->days : 1;
}

/**
 * Determines the status of a booking based on its dates.
 * @param string $arrival_date
 * @param string $departure_date
 * @return array An array containing the status text and a corresponding CSS class.
 */
function getBookingStatus($arrival_date, $departure_date) {
    $today = new DateTime();
    $arrival = new DateTime($arrival_date);
    $departure = new DateTime($departure_date);

    $today->setTime(0, 0, 0);
    $arrival->setTime(0, 0, 0);
    $departure->setTime(0, 0, 0);

    if ($today < $arrival) {
        return ['text' => 'Upcoming', 'class' => 'bg-emerald-100 text-emerald-800'];
    } elseif ($today >= $arrival && $today < $departure) {
        return ['text' => 'Ongoing', 'class' => 'bg-blue-100 text-blue-800'];
    } else {
        return ['text' => 'Completed', 'class' => 'bg-gray-200 text-gray-800'];
    }
}


// --- FETCH USER'S BOOKINGS ---
$user_id = $_SESSION['userid'];
$bookings = [];

$sql = "SELECT 
            b.booking_id, b.arrival_date, b.departure_date, b.adults, b.children, b.total_price,
            p.id as property_id, p.header, p.location, p.image1
        FROM bookings b
        JOIN properties p ON b.property_id = p.id
        WHERE b.user_id = ?
        ORDER BY b.arrival_date DESC";

$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();

while ($row = $result->fetch_assoc()) {
    $bookings[] = $row;
}

$stmt->close();
$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Bookings - Floracion Hospitality</title>
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Playfair+Display:wght@700;800&family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">

    <style>
        body { font-family: 'Poppins', sans-serif; background-color: #f8fafc; color: #1e293b; overflow-x: hidden; }
        .font-serif { font-family: 'Playfair Display', serif; }
        .gsap-reveal, .ticket-card, .no-bookings-card {
            opacity: 0;
            visibility: hidden;
        }
        .ticket-card {
            transition: all 0.4s cubic-bezier(0.25, 0.46, 0.45, 0.94);
            border-left: 5px solid #10b981;
        }
        .ticket-card:hover {
            transform: translateY(-8px) scale(1.02);
            box-shadow: 0 20px 40px -15px rgba(0, 0, 0, 0.15);
        }
        .ticket-card.status-upcoming { border-left-color: #10b981; }
        .ticket-card.status-ongoing { border-left-color: #3b82f6; }
        .ticket-card.status-completed { border-left-color: #6b7280; }
        .ticket-image-link::after {
            content: 'View Property';
            position: absolute; inset: 0;
            background-color: rgba(0,0,0,0.6);
            color: white;
            display: flex; align-items: center; justify-content: center;
            font-weight: 600; opacity: 0;
            transition: opacity 0.3s ease;
        }
        .ticket-image-link:hover::after { opacity: 1; }
    </style>
</head>
<body>

    <!-- Animated Page Transition Panel -->
    <div id="animation-panel" class="fixed top-0 left-0 w-full h-full bg-emerald-600 z-[100]"></div>

    <?php include 'navbar.php'; ?>

    <main class="main-container">
        <!-- HERO SECTION -->
        <section class="hero-section relative bg-gray-800 flex items-center justify-center text-white overflow-hidden py-24 md:py-32">
            <div class="absolute inset-0 w-full h-full z-0">
                <img src="https://images.unsplash.com/photo-1520250497591-112f2f40a3f4?q=80&w=2070&auto=format&fit=crop" class="absolute inset-0 w-full h-full object-cover opacity-20" alt="Lobby background">
                <div class="absolute inset-0 bg-gradient-to-t from-gray-900 via-gray-900/70 to-transparent"></div>
            </div>
            <div class="container mx-auto px-6 relative text-center z-10">
                <div class="hero-content">
                    <span class="gsap-reveal hero-subtitle block text-emerald-400 font-semibold uppercase tracking-widest">Your Personal Itinerary</span>
                    <h1 class="gsap-reveal hero-title font-serif text-5xl md:text-7xl mt-2 mb-4">My Bookings</h1>
                </div>
            </div>
        </section>

        <!-- BOOKINGS LISTING SECTION -->
        <section class="bookings-listing py-16 lg:py-24">
            <div class="container mx-auto px-6">
                
                <?php if (empty($bookings)): ?>
                    <!-- NO BOOKINGS FOUND VIEW -->
                    <div class="no-bookings-card text-center max-w-lg mx-auto">
                        <div class="text-7xl text-emerald-400 mb-6"><i class="fas fa-suitcase-rolling"></i></div>
                        <h2 class="font-serif text-4xl text-gray-800 mb-3">No Journeys Yet</h2>
                        <p class="text-gray-500 mb-8">Your travel history is empty. It's the perfect time to plan your next getaway!</p>
                        <a href="index.php" class="inline-block bg-emerald-600 text-white font-bold py-3 px-8 rounded-lg text-lg transition-transform duration-300 hover:bg-emerald-700 hover:scale-105">
                            <i class="fas fa-search mr-2"></i> Explore Stays
                        </a>
                    </div>
                <?php else: ?>
                    <!-- BOOKINGS FOUND VIEW -->
                    <div class="space-y-10">
                        <?php foreach ($bookings as $booking): ?>
                            <?php $status = getBookingStatus($booking['arrival_date'], $booking['departure_date']); ?>
                            <div class="ticket-card bg-white rounded-xl shadow-lg overflow-hidden flex flex-col md:flex-row <?php echo 'status-' . strtolower($status['text']); ?>">
                                <!-- Property Image -->
                                <div class="md:w-1/3 xl:w-1/4 flex-shrink-0">
                                    <a href="property_details.php?id=<?php echo $booking['property_id']; ?>" class="ticket-image-link block h-60 md:h-full relative">
                                        <img src="../owner/<?php echo htmlspecialchars($booking['image1']); ?>" alt="<?php echo htmlspecialchars($booking['header']); ?>" class="w-full h-full object-cover">
                                    </a>
                                </div>
                                
                                <!-- Booking Details -->
                                <div class="p-6 md:p-8 flex-grow flex flex-col justify-between">
                                    <div>
                                        <div class="flex flex-col sm:flex-row justify-between sm:items-start mb-4">
                                            <div>
                                                <h3 class="font-serif text-2xl lg:text-3xl font-bold text-gray-800"><?php echo htmlspecialchars($booking['header']); ?></h3>
                                                <p class="text-gray-500 flex items-center gap-2 mt-1"><i class="fas fa-map-marker-alt text-sm"></i><?php echo htmlspecialchars($booking['location']); ?></p>
                                            </div>
                                            <div class="mt-3 sm:mt-0 flex-shrink-0">
                                                <span class="font-semibold py-1 px-3 rounded-full text-sm <?php echo $status['class']; ?>"><i class="fas fa-info-circle mr-1.5"></i><?php echo $status['text']; ?></span>
                                            </div>
                                        </div>
                                        
                                        <div class="grid grid-cols-2 gap-x-6 gap-y-5 text-gray-600 border-t border-gray-200 pt-5">
                                            <div class="flex items-center gap-3"><i class="fas fa-calendar-alt text-xl text-emerald-500"></i><div><span class="text-xs uppercase font-semibold text-gray-400">Arrival</span><p class="font-medium text-gray-800"><?php echo date("D, M j, Y", strtotime($booking['arrival_date'])); ?></p></div></div>
                                            <div class="flex items-center gap-3"><i class="fas fa-calendar-check text-xl text-emerald-500"></i><div><span class="text-xs uppercase font-semibold text-gray-400">Departure</span><p class="font-medium text-gray-800"><?php echo date("D, M j, Y", strtotime($booking['departure_date'])); ?></p></div></div>
                                            <div class="flex items-center gap-3"><i class="fas fa-moon text-xl text-emerald-500"></i><div><span class="text-xs uppercase font-semibold text-gray-400">Nights</span><p class="font-medium text-gray-800"><?php echo calculate_nights($booking['arrival_date'], $booking['departure_date']); ?> Nights</p></div></div>
                                            <div class="flex items-center gap-3"><i class="fas fa-users text-xl text-emerald-500"></i><div><span class="text-xs uppercase font-semibold text-gray-400">Guests</span><p class="font-medium text-gray-800"><?php echo $booking['adults']; ?> Adults, <?php echo $booking['children']; ?> Children</p></div></div>
                                        </div>
                                    </div>

                                    <div class="border-t border-dashed border-gray-300 mt-6 pt-5 flex flex-col sm:flex-row justify-between items-center">
                                        <p class="text-gray-500 text-sm flex items-center gap-2"><i class="fas fa-receipt"></i>Booking ID: <strong class="text-gray-700"><?php echo htmlspecialchars($booking['booking_id']); ?></strong></p>
                                        <p class="font-serif text-2xl text-emerald-600 font-bold mt-3 sm:mt-0">Total: ₹<?php echo number_format($booking['total_price']); ?></p>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </section>
    </main>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/gsap/3.12.5/gsap.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/gsap/3.12.5/ScrollTrigger.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', () => {
            gsap.registerPlugin(ScrollTrigger);

            const masterTl = gsap.timeline();

            // 1. Page transition animation
            masterTl.to('#animation-panel', {
                xPercent: 100,
                duration: 1.2,
                ease: 'power4.inOut'
            }).set('#animation-panel', { display: 'none' });

            // 2. Hero text animation
            masterTl.to('.gsap-reveal', {
                opacity: 1,
                visibility: 'visible',
                y: 0,
                duration: 0.8,
                stagger: 0.2,
                ease: 'power3.out'
            }, "-=0.8");

            // 3. Animate booking cards on scroll
            gsap.utils.toArray('.ticket-card').forEach(card => {
                gsap.fromTo(card, 
                    { y: 60, opacity: 0, visibility: 'hidden' }, 
                    {
                        y: 0,
                        opacity: 1,
                        visibility: 'visible',
                        duration: 1,
                        ease: 'power3.out',
                        scrollTrigger: {
                            trigger: card,
                            start: 'top 85%',
                            toggleActions: 'play none none none',
                            once: true
                        }
                    }
                );
            });
            
            // 4. Animate "No Bookings" message if it exists
            if(document.querySelector('.no-bookings-card')) {
                 gsap.fromTo('.no-bookings-card', 
                    { y: 50, opacity: 0 },
                    {
                        y: 0,
                        opacity: 1,
                        visibility: 'visible',
                        duration: 1,
                        delay: 0.5,
                        ease: 'power3.out'
                    }
                );
            }
        });
    </script>

</body>
</html>
