<?php
// FILE: dashboard.php (New Stylish Version)

// 1. INITIALIZE SESSION & AUTHENTICATION
session_start();

// Redirect to login if not logged in
if (!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) {
    header("location: owner_login.php");
    exit;
}

// 2. INCLUDE DATABASE CONNECTION
require_once "../config/db.php"; 

// 3. FETCH DATA FOR STATS
$owner_id = $_SESSION['id']; 

// Stat 1: Total Properties
$sql_total = "SELECT COUNT(id) as total_count FROM properties WHERE owner_id = ?";
$stmt_total = $link->prepare($sql_total);
$stmt_total->bind_param("i", $owner_id);
$stmt_total->execute();
$result_total = $stmt_total->get_result();
$total_properties = $result_total->fetch_assoc()['total_count'] ?? 0;
$stmt_total->close();

// Stat 2: Active Properties
$sql_active = "SELECT COUNT(id) as active_count FROM properties WHERE owner_id = ? AND status = 'active'";
$stmt_active = $link->prepare($sql_active);
$stmt_active->bind_param("i", $owner_id);
$stmt_active->execute();
$result_active = $stmt_active->get_result();
$active_properties = $result_active->fetch_assoc()['active_count'] ?? 0;
$stmt_active->close();

// Stat 3: Inactive Properties
$inactive_properties = $total_properties - $active_properties;

// Stat 4: Unique Locations
$sql_locations = "SELECT COUNT(DISTINCT location) as location_count FROM properties WHERE owner_id = ?";
$stmt_locations = $link->prepare($sql_locations);
$stmt_locations->bind_param("i", $owner_id);
$stmt_locations->execute();
$result_locations = $stmt_locations->get_result();
$unique_locations = $result_locations->fetch_assoc()['location_count'] ?? 0;
$stmt_locations->close();

// New: Fetch Recent Properties
$recent_properties = [];
$sql_recent = "SELECT header, location, image1, status FROM properties WHERE owner_id = ? ORDER BY upload_date DESC LIMIT 3";
$stmt_recent = $link->prepare($sql_recent);
$stmt_recent->bind_param("i", $owner_id);
$stmt_recent->execute();
$result_recent = $stmt_recent->get_result();
while ($row = $result_recent->fetch_assoc()) {
    $recent_properties[] = $row;
}
$stmt_recent->close();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Owner Dashboard</title>
    
    <!-- Libraries -->
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/gsap/3.12.5/gsap.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">
    
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800&display=swap');
        body {
            font-family: 'Inter', sans-serif;
            background-color: #f1f5f9; /* Light Slate Gray */
        }
        #star-background {
            position: fixed; top: 0; left: 0;
            width: 100%; height: 100%; z-index: -1;
        }
        .stat-card {
            border-radius: 1.5rem; /* More rounded */
            padding: 1.5rem;
            color: white;
            position: relative;
            overflow: hidden;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        .stat-card:hover {
            transform: translateY(-8px);
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
        }
        .stat-card .icon-bg {
            position: absolute;
            bottom: -1rem;
            right: -1rem;
            font-size: 6rem;
            opacity: 0.15;
            transform: rotate(-15deg);
            transition: transform 0.4s ease;
        }
        .stat-card:hover .icon-bg {
            transform: rotate(0deg) scale(1.1);
        }
        .chart-container, .recent-listings {
            background-color: white;
            border-radius: 1.5rem;
            padding: 2rem;
            box-shadow: 0 10px 15px -3px rgb(0 0 0 / 0.07), 0 4px 6px -4px rgb(0 0 0 / 0.07);
        }
    </style>
</head>
<body class="text-gray-800">
    <canvas id="star-background"></canvas>
    <div class="p-4 sm:p-6 lg:p-8">
        <div class="max-w-7xl mx-auto">
            
            <!-- Header -->
            <header data-aos="fade-down" class="mb-10">
                <h1 class="text-4xl font-extrabold text-gray-900">Hi <?php echo htmlspecialchars(explode(' ', $_SESSION["name"])[0]); ?>, Welcome Back! 👋</h1>
                <p class="text-lg text-gray-500 mt-2">Here's a quick overview of your property portfolio.</p>
            </header>

            <!-- Stats Grid -->
            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-10">
                <!-- Total Properties -->
                <div class="stat-card bg-gradient-to-br from-purple-500 to-indigo-600" data-aos="fade-up">
                    <div class="relative z-10">
                        <p class="font-medium">Total Properties</p>
                        <p class="text-5xl font-extrabold mt-1"><?php echo $total_properties; ?></p>
                        <p class="text-sm opacity-80 mt-2">Across all locations</p>
                    </div>
                    <i class="fas fa-hotel icon-bg"></i>
                </div>
                <!-- Active Properties -->
                <div class="stat-card bg-gradient-to-br from-green-500 to-teal-600" data-aos="fade-up" data-aos-delay="100">
                     <div class="relative z-10">
                        <p class="font-medium">Active Properties</p>
                        <p class="text-5xl font-extrabold mt-1"><?php echo $active_properties; ?></p>
                        <p class="text-sm opacity-80 mt-2">Currently available</p>
                    </div>
                    <i class="fas fa-door-open icon-bg"></i>
                </div>
                <!-- Inactive Properties -->
                <div class="stat-card bg-gradient-to-br from-amber-500 to-orange-600" data-aos="fade-up" data-aos-delay="200">
                    <div class="relative z-10">
                        <p class="font-medium">Inactive Properties</p>
                        <p class="text-5xl font-extrabold mt-1"><?php echo $inactive_properties; ?></p>
                        <p class="text-sm opacity-80 mt-2">Currently booked</p>
                    </div>
                    <i class="fas fa-door-closed icon-bg"></i>
                </div>
                <!-- Unique Locations -->
                <div class="stat-card bg-gradient-to-br from-pink-500 to-rose-600" data-aos="fade-up" data-aos-delay="300">
                    <div class="relative z-10">
                        <p class="font-medium">Unique Locations</p>
                        <p class="text-5xl font-extrabold mt-1"><?php echo $unique_locations; ?></p>
                        <p class="text-sm opacity-80 mt-2">Cities & regions</p>
                    </div>
                    <i class="fas fa-map-marked-alt icon-bg"></i>
                </div>
            </div>

            <!-- Chart & Recent Listings Grid -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <div class="chart-container lg:col-span-2" data-aos="fade-up" data-aos-delay="400">
                    <h2 class="text-2xl font-bold mb-4">Properties Overview</h2>
                    <div style="height: 400px;">
                        <canvas id="propertyStatusChart"></canvas>
                    </div>
                </div>
                <div class="recent-listings" data-aos="fade-up" data-aos-delay="500">
                    <h2 class="text-2xl font-bold mb-4">Recent Listings</h2>
                    <div class="space-y-4">
                        <?php if (!empty($recent_properties)): ?>
                            <?php foreach($recent_properties as $prop): ?>
                                <div class="flex items-center space-x-4">
                                    <img src="<?php echo htmlspecialchars($prop['image1']); ?>" alt="property" class="w-16 h-16 rounded-lg object-cover">
                                    <div>
                                        <p class="font-semibold text-gray-800"><?php echo htmlspecialchars($prop['header']); ?></p>
                                        <p class="text-sm text-gray-500"><?php echo htmlspecialchars($prop['location']); ?></p>
                                    </div>
                                    <span class="text-xs font-bold py-1 px-2 rounded-full ml-auto <?php echo $prop['status'] === 'active' ? 'bg-green-100 text-green-700' : 'bg-amber-100 text-amber-700'; ?>">
                                        <?php echo ucfirst($prop['status']); ?>
                                    </span>
                                </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <p class="text-gray-500">No recent properties found.</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script>
        AOS.init({ duration: 800, once: true });

        // GSAP Shining Stars Background
        const canvas = document.getElementById('star-background');
        const ctx = canvas.getContext('2d');
        let stars = [];
        const numStars = 150;

        function resizeCanvas() {
            canvas.width = window.innerWidth;
            canvas.height = window.innerHeight;
        }
        resizeCanvas();
        window.addEventListener('resize', resizeCanvas);

        for (let i = 0; i < numStars; i++) {
            stars.push({
                x: Math.random() * canvas.width, y: Math.random() * canvas.height,
                radius: Math.random() * 1.2,
                vx: (Math.random() - 0.5) * 0.1, vy: (Math.random() - 0.5) * 0.1,
                opacity: Math.random() * 0.5 + 0.1
            });
        }

        function drawStars() {
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            stars.forEach(star => {
                ctx.beginPath();
                ctx.arc(star.x, star.y, star.radius, 0, Math.PI * 2);
                ctx.fillStyle = `rgba(100, 116, 139, ${star.opacity})`;
                ctx.fill();
            });
        }

        function updateStars() {
            stars.forEach(star => {
                star.x += star.vx; star.y += star.vy;
                if (star.x < 0 || star.x > canvas.width) star.vx = -star.vx;
                if (star.y < 0 || star.y > canvas.height) star.vy = -star.vy;
            });
        }
        gsap.ticker.add(() => { updateStars(); drawStars(); });

        // Chart.js Doughnut Chart
        const chartCtx = document.getElementById('propertyStatusChart').getContext('2d');
        new Chart(chartCtx, {
            type: 'doughnut',
            data: {
                labels: ['Active', 'Inactive'],
                datasets: [{
                    data: [<?php echo $active_properties; ?>, <?php echo $inactive_properties; ?>],
                    backgroundColor: ['#10b981', '#f59e0b'],
                    borderColor: '#ffffff',
                    borderWidth: 4,
                    hoverOffset: 15
                }]
            },
            options: {
                responsive: true, maintainAspectRatio: false, cutout: '75%',
                plugins: {
                    legend: {
                        position: 'bottom',
                        labels: { padding: 25, font: { size: 14, family: "'Inter', sans-serif" } }
                    }
                }
            }
        });
    </script>
</body>
</html>
