<?php
// FILE: add_property.php (Updated with Rate per Day)

// Initialize the session
session_start();

// Check if the user is logged in, if not then redirect to the login page
if(!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true){
    header("location: owner_login.php");
    exit;
}

// Include the database connection file
require_once "../config/db.php";

// Initialize variables
$feedback_message = "";
$feedback_class = "";

// Process form data when form is submitted
if($_SERVER["REQUEST_METHOD"] == "POST"){

    // --- FORM DATA VALIDATION ---
    $property_type = $_POST['property_type'] ?? '';
    $rate = $_POST['rate'] ?? ''; // New rate field
    $location = trim($_POST['location'] ?? '');
    $header = trim($_POST['header'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $rules = trim($_POST['rules'] ?? '');
    $owner_id = $_SESSION['id']; // Get the logged-in owner's ID

    // Basic validation
    if(empty($property_type) || empty($location) || empty($rate) || empty($header) || empty($description)){
        $feedback_message = "Please fill in all mandatory fields.";
        $feedback_class = "feedback-error";
    } elseif (!is_numeric($rate) || $rate < 0) {
        $feedback_message = "Please enter a valid, non-negative rate.";
        $feedback_class = "feedback-error";
    } elseif (empty($_FILES["image1"]["name"])) {
        $feedback_message = "The first image (Main Image) is mandatory.";
        $feedback_class = "feedback-error";
    } else {
        
        // --- IMAGE UPLOAD HANDLING ---
        $upload_dir = "uploads/";
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0777, true);
        }
        $image_paths = [];
        $allowed_types = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
        $has_error = false;

        for ($i = 1; $i <= 4; $i++) {
            $file_key = "image" . $i;
            if (!empty($_FILES[$file_key]["name"])) {
                $file_name = time() . '_' . basename($_FILES[$file_key]["name"]);
                $target_file = $upload_dir . $file_name;
                $file_type = strtolower(pathinfo($target_file, PATHINFO_EXTENSION));

                $check = getimagesize($_FILES[$file_key]["tmp_name"]);
                if($check === false) {
                    $feedback_message = "File '{$_FILES[$file_key]['name']}' is not a valid image.";
                    $feedback_class = "feedback-error";
                    $has_error = true;
                    break;
                }

                if(!in_array($file_type, $allowed_types)) {
                    $feedback_message = "Invalid file type for '{$_FILES[$file_key]['name']}'. Only JPG, PNG, GIF, WEBP allowed.";
                    $feedback_class = "feedback-error";
                    $has_error = true;
                    break;
                }

                if (move_uploaded_file($_FILES[$file_key]["tmp_name"], $target_file)) {
                    $image_paths[$file_key] = $target_file;
                } else {
                    $feedback_message = "Sorry, there was an error uploading your file.";
                    $feedback_class = "feedback-error";
                    $has_error = true;
                    break;
                }
            } else {
                $image_paths[$file_key] = null;
            }
        }

        // If all uploads were successful, insert into database
        if(!$has_error) {
            // Updated SQL query with the new 'rate' column
            $sql = "INSERT INTO properties (owner_id, property_type, rate, location, header, description, rules, image1, image2, image3, image4) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

            if($stmt = mysqli_prepare($link, $sql)){
                // Updated bind_param: 'd' for double/decimal type for rate
                mysqli_stmt_bind_param($stmt, "isdssssssss", 
                    $owner_id, 
                    $property_type, 
                    $rate, // Bind the new rate variable
                    $location, 
                    $header, 
                    $description, 
                    $rules, 
                    $image_paths['image1'], 
                    $image_paths['image2'], 
                    $image_paths['image3'], 
                    $image_paths['image4']
                );

                if(mysqli_stmt_execute($stmt)){
                    $feedback_message = "Property listed successfully! It's now live.";
                    $feedback_class = "feedback-success";
                } else {
                    $feedback_message = "Database error. Please try again later.";
                    $feedback_class = "feedback-error";
                }
                mysqli_stmt_close($stmt);
            }
        }
    }
    mysqli_close($link);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add New Property - Hospitality Owner</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/gsap/3.12.5/gsap.min.js"></script>
    <script src="https://unpkg.com/lucide@latest"></script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Playfair+Display:wght@700&family=Inter:wght@400;500;600&display=swap');
        :root {
            --primary-color: #2c3e50; /* Deep Slate Blue */
            --secondary-color: #c0a080; /* Soft Gold */
            --background-color: #f7f9fc; /* Very Light Blue-Gray */
            --form-background: #ffffff;
            --text-color: #34495e;
            --border-color: #dfe4ea;
            --success-color: #10b981;
            --error-color: #ef4444;
        }
        body {
            font-family: 'Inter', sans-serif;
            background-color: var(--background-color);
            color: var(--text-color);
        }
        .form-header {
            font-family: 'Playfair Display', serif;
        }
        .form-container {
            background: var(--form-background);
            border: 1px solid var(--border-color);
        }
        .form-input, .form-select, .form-textarea {
            background-color: #fdfdfd;
            border: 1px solid var(--border-color);
            transition: all 0.3s ease;
            border-radius: 8px;
        }
        .form-input:focus, .form-select:focus, .form-textarea:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(44, 62, 80, 0.1);
        }
        .btn-primary {
            background-color: var(--primary-color);
            transition: all 0.3s ease;
        }
        .btn-primary:hover {
            background-color: #34495e;
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(0,0,0,0.1);
        }
        .image-upload-box {
            border: 2px dashed var(--border-color);
            transition: all 0.3s ease;
        }
        .image-upload-box:hover {
            border-color: var(--primary-color);
            background-color: #f7f9fc;
        }
        .feedback-success {
            background-color: rgba(16, 185, 129, 0.1);
            color: var(--success-color);
            border-left: 4px solid var(--success-color);
        }
        .feedback-error {
            background-color: rgba(239, 68, 68, 0.1);
            color: var(--error-color);
            border-left: 4px solid var(--error-color);
        }
    </style>
</head>
<body>
    <div class="min-h-screen w-full p-4 md:p-8 flex items-center justify-center">
        <div class="w-full max-w-5xl mx-auto">
            
            <div class="form-container rounded-2xl shadow-2xl p-8 md:p-12" id="property-form">
                <div class="text-center mb-10">
                    <i data-lucide="building" class="mx-auto h-12 w-12" style="color: var(--primary-color)"></i>
                    <h1 class="text-4xl md:text-5xl font-bold form-header mt-4" style="color: var(--primary-color)">List a New Property</h1>
                    <p class="text-gray-500 mt-2">Fill in the details below to showcase your property to the world.</p>
                </div>

                <?php if(!empty($feedback_message)): ?>
                    <div id="feedback-message" class="text-center p-4 mb-8 rounded-lg font-semibold <?php echo $feedback_class; ?>">
                        <?php echo $feedback_message; ?>
                    </div>
                <?php endif; ?>

                <form action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="post" enctype="multipart/form-data" class="space-y-8">
                    
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-8 form-section">
                        <div>
                            <label for="property_type" class="block text-sm font-medium text-gray-700 mb-2 flex items-center"><i data-lucide="building-2" class="h-4 w-4 mr-2"></i>Property Type <span class="text-red-500 ml-1">*</span></label>
                            <select id="property_type" name="property_type" class="form-select w-full p-3" required>
                                <option value="" disabled selected>Select a property type</option>
                                <option value="Hotel">Hotel</option>
                                <option value="Resort">Resort</option>
                                <option value="Homestay">Homestay</option>
                                <option value="Villa">Villa</option>
                            </select>
                        </div>
                        <div>
                            <label for="rate" class="block text-sm font-medium text-gray-700 mb-2 flex items-center"><i data-lucide="indian-rupee" class="h-4 w-4 mr-2"></i>Rate per Day (₹) <span class="text-red-500 ml-1">*</span></label>
                            <input type="number" id="rate" name="rate" placeholder="e.g., 3000" class="form-input w-full p-3" required min="0" step="1">
                        </div>
                        <div>
                            <label for="location" class="block text-sm font-medium text-gray-700 mb-2 flex items-center"><i data-lucide="map-pin" class="h-4 w-4 mr-2"></i>Location <span class="text-red-500 ml-1">*</span></label>
                            <input type="text" id="location" name="location" placeholder="e.g., Dhanbad, Jharkhand" class="form-input w-full p-3" required>
                        </div>
                    </div>
                    
                    <div class="form-section">
                        <label for="header" class="block text-sm font-medium text-gray-700 mb-2 flex items-center"><i data-lucide="type" class="h-4 w-4 mr-2"></i>Property Header <span class="text-red-500 ml-1">*</span></label>
                        <input type="text" id="header" name="header" placeholder="e.g., 'Luxurious Riverside Villa with Pool'" class="form-input w-full p-3" required>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-8 form-section">
                        <div>
                            <label for="description" class="block text-sm font-medium text-gray-700 mb-2 flex items-center"><i data-lucide="file-text" class="h-4 w-4 mr-2"></i>Description <span class="text-red-500 ml-1">*</span></label>
                            <textarea id="description" name="description" rows="6" placeholder="Describe what makes your property special..." class="form-textarea w-full p-3" required></textarea>
                        </div>
                        <div>
                            <label for="rules" class="block text-sm font-medium text-gray-700 mb-2 flex items-center"><i data-lucide="gavel" class="h-4 w-4 mr-2"></i>Rules & Regulations (Optional)</label>
                            <textarea id="rules" name="rules" rows="6" placeholder="e.g., 'No pets allowed. No smoking inside. Quiet hours after 10 PM.'" class="form-textarea w-full p-3"></textarea>
                        </div>
                    </div>

                    <div class="form-section">
                        <label class="block text-sm font-medium text-gray-700 mb-2 flex items-center"><i data-lucide="image" class="h-4 w-4 mr-2"></i>Property Images</label>
                        <div class="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-4 gap-6">
                            <div class="image-upload-box p-4 rounded-lg text-center">
                                <label for="image1" class="cursor-pointer">
                                    <i data-lucide="upload-cloud" class="h-10 w-10 mx-auto text-gray-400"></i>
                                    <span class="mt-2 block text-sm text-gray-600">Main Image <span class="text-red-500">*</span></span>
                                    <span class="mt-1 block text-xs text-gray-500" id="file-name1">Click to upload</span>
                                </label>
                                <input type="file" id="image1" name="image1" class="hidden" accept="image/*" onchange="updateFileName(this.id)">
                            </div>
                             <?php for ($i = 2; $i <= 4; $i++): ?>
                            <div class="image-upload-box p-4 rounded-lg text-center">
                                <label for="image<?php echo $i; ?>" class="cursor-pointer">
                                    <i data-lucide="plus-square" class="h-10 w-10 mx-auto text-gray-400"></i>
                                    <span class="mt-2 block text-sm text-gray-600">Optional Image <?php echo $i; ?></span>
                                    <span class="mt-1 block text-xs text-gray-500" id="file-name<?php echo $i; ?>">Click to upload</span>
                                </label>
                                <input type="file" id="image<?php echo $i; ?>" name="image<?php echo $i; ?>" class="hidden" accept="image/*" onchange="updateFileName(this.id)">
                            </div>
                            <?php endfor; ?>
                        </div>
                    </div>

                    <div class="pt-4 form-section">
                        <button type="submit" class="w-full btn-primary text-white font-bold py-4 px-4 rounded-lg text-lg">
                            List My Property
                        </button>
                    </div>
                </form>
            </div>
             <p class="text-center mt-8 text-sm text-gray-500">
                <a href="index.php" class="hover:underline flex items-center justify-center group">
                    <i data-lucide="arrow-left" class="h-4 w-4 mr-2 transition-transform group-hover:-translate-x-1"></i>
                    Back to Dashboard
                </a>
            </p>
        </div>
    </div>
    <script>
        lucide.createIcons();

        // GSAP Animations
        gsap.from("#property-form", { duration: 1, opacity: 0, y: 80, ease: "power3.out" });
        gsap.from(".form-section", { duration: 0.8, opacity: 0, x: -40, stagger: 0.15, ease: "power2.out", delay: 0.5 });
        
        const feedbackMessage = document.getElementById('feedback-message');
        if (feedbackMessage) {
            gsap.fromTo(feedbackMessage, {opacity: 0, y: 20}, {opacity: 1, y: 0, duration: 0.5, ease: 'power2.out'});
        }

        // Function to update file name display
        function updateFileName(inputId) {
            const input = document.getElementById(inputId);
            const file = input.files[0];
            const displayId = 'file-name' + inputId.slice(-1);
            const displaySpan = document.getElementById(displayId);
            if (file) {
                displaySpan.textContent = file.name.length > 15 ? file.name.substring(0, 12) + '...' : file.name;
                displaySpan.style.color = 'var(--success-color)';
                displaySpan.style.fontWeight = '600';
            } else {
                displaySpan.textContent = 'Click to upload';
                displaySpan.style.color = '';
                displaySpan.style.fontWeight = '';
            }
        }
    </script>
</body>
</html>