<?php
// FILE: view_properties.php (Enhanced to Show and Edit Rates)

// 1. INCLUDE DATABASE CONNECTION
require_once '../config/db.php';

// 2. API LOGIC HANDLER
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    
    header('Content-Type: application/json');
    $response = ['success' => false, 'message' => 'Invalid request.'];

    // --- HANDLE DELETE ACTION ---
    if ($_POST['action'] === 'delete') {
        $propertyId = isset($_POST['id']) ? intval($_POST['id']) : 0;
        if ($propertyId > 0) {
            $stmt = $link->prepare("SELECT image1, image2, image3, image4 FROM properties WHERE id = ?");
            $stmt->bind_param("i", $propertyId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                foreach ($row as $image_path) {
                    if (!empty($image_path) && file_exists($image_path)) {
                        unlink($image_path);
                    }
                }
            }
            $stmt->close();

            $deleteStmt = $link->prepare("DELETE FROM properties WHERE id = ?");
            $deleteStmt->bind_param("i", $propertyId);
            if ($deleteStmt->execute()) {
                $response = ['success' => true, 'message' => 'Property deleted successfully.'];
            } else {
                $response = ['success' => false, 'message' => 'Failed to delete property record.'];
            }
            $deleteStmt->close();
        }
    }

    // --- HANDLE UPDATE ACTION ---
    if ($_POST['action'] === 'update') {
        $propertyId = isset($_POST['id']) ? intval($_POST['id']) : 0;
        $header = $_POST['header'] ?? '';
        $rate = $_POST['rate'] ?? 0.0; // Added rate
        $description = $_POST['description'] ?? '';
        $rules = $_POST['rules'] ?? '';
        $status = $_POST['status'] ?? 'inactive';

        if ($propertyId > 0 && is_numeric($rate) && $rate >= 0) {
            $stmt = $link->prepare("UPDATE properties SET header = ?, rate = ?, description = ?, rules = ?, status = ? WHERE id = ?");
            // Updated bind_param with 'd' for rate (double)
            $stmt->bind_param("sdsssi", $header, $rate, $description, $rules, $status, $propertyId);

            if ($stmt->execute()) {
                // Fetch updated data to send back to the client
                $selectStmt = $link->prepare("SELECT header, rate, description, rules, status, property_type FROM properties WHERE id = ?");
                $selectStmt->bind_param("i", $propertyId);
                $selectStmt->execute();
                $updatedData = $selectStmt->get_result()->fetch_assoc();
                $selectStmt->close();
                $response = ['success' => true, 'message' => 'Property updated.', 'updatedData' => $updatedData];
            } else {
                $response = ['success' => false, 'message' => 'Failed to update property.'];
            }
            $stmt->close();
        } else {
             $response = ['success' => false, 'message' => 'Invalid rate provided.'];
        }
    }

    echo json_encode($response);
    $link->close();
    exit;
}

// 3. PAGE RENDERING LOGIC - Added 'rate' to the query
$sql = "SELECT id, property_type, rate, header, description, rules, status, image1, image2, image3, image4, upload_date FROM properties ORDER BY upload_date DESC";
$result = $link->query($sql);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Properties</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/swiper@11/swiper-bundle.min.css" />
    <style>
        :root {
            --primary-color: #0d9488; /* Teal */
            --secondary-color: #4338ca; /* Indigo */
            --background-color: #f8fafc; /* Lighter Gray */
            --card-background: #ffffff;
            --text-color: #1e293b; /* Slate */
            --light-text-color: #64748b;
            --shadow: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
            --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
            --status-active: #16a34a; /* Green */
            --status-inactive: #dc2626; /* Red */
        }
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Inter', sans-serif;
            background-color: var(--background-color);
            color: var(--text-color);
            overflow-x: hidden;
        }
        .page-container { padding: 2rem 5%; position: relative; }
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 3rem;
        }
        .header-title { font-size: 2.2rem; font-weight: 700; }
        .add-property-btn {
            display: inline-flex; align-items: center; gap: 10px;
            padding: 12px 24px; font-size: 0.95rem; font-weight: 600;
            color: #fff; background: var(--primary-color);
            border: none; border-radius: 8px; cursor: pointer; text-decoration: none;
            box-shadow: var(--shadow); transition: all 0.3s ease;
        }
        .add-property-btn:hover { transform: translateY(-3px); box-shadow: var(--shadow-lg); background: #0f766e; }
        .property-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(340px, 1fr));
            gap: 2.5rem;
        }
        .property-card {
            background: var(--card-background); border-radius: 16px;
            overflow: hidden; box-shadow: var(--shadow);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            display: flex; flex-direction: column;
            border: 1px solid #e2e8f0;
        }
        .property-card:hover { transform: translateY(-8px); box-shadow: var(--shadow-lg); }
        .card-slider-wrapper { position: relative; }
        .card-slider { width: 100%; height: 220px; background-color: #e2e8f0; }
        .swiper { width: 100%; height: 100%; }
        .swiper-slide img { width: 100%; height: 100%; object-fit: cover; }
        .status-badge {
            position: absolute; top: 1rem; left: 1rem; z-index: 10;
            padding: 4px 12px; border-radius: 9999px; font-size: 0.75rem;
            font-weight: 600; color: #fff; text-transform: uppercase;
            background-color: rgba(0, 0, 0, 0.3);
        }
        .card-content { padding: 1.25rem; flex-grow: 1; display: flex; flex-direction: column; }
        .card-header { font-size: 1.25rem; font-weight: 600; margin-bottom: 0.25rem; }
        .card-property-type { font-size: 0.9rem; font-weight: 600; color: var(--primary-color); margin-bottom: 0.75rem; }
        /* New Style for the Rate Display */
        .card-rate {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--text-color);
            margin-bottom: 1.25rem;
        }
        .card-rate i { color: var(--light-text-color); font-size: 1.1rem; margin-right: 5px;}
        .card-rate span { font-size: 0.9rem; font-weight: 400; color: var(--light-text-color); }

        .detail-section { margin-bottom: 1rem; }
        .detail-section h4 { font-size: 0.8rem; text-transform: uppercase; letter-spacing: 0.5px; font-weight: 600; color: var(--light-text-color); margin-bottom: 0.5rem; }
        .detail-section p {
            font-size: 0.9rem; line-height: 1.6; color: var(--text-color);
            display: -webkit-box; -webkit-line-clamp: 2;
            -webkit-box-orient: vertical; overflow: hidden;
        }
        .card-footer {
            border-top: 1px solid #e2e8f0; padding: 0.75rem 1.25rem;
            margin-top: auto; font-size: 0.8rem; color: var(--light-text-color);
            display: flex; justify-content: space-between; align-items: center;
        }
        .card-menu { position: relative; }
        .menu-toggle {
            background: transparent; color: var(--light-text-color); border: none;
            width: 35px; height: 35px; border-radius: 50%;
            cursor: pointer; font-size: 1rem; display: flex;
            justify-content: center; align-items: center; transition: background 0.3s, color 0.3s;
        }
        .menu-toggle:hover { background: #f1f5f9; color: var(--text-color); }
        .menu-options {
            position: absolute; bottom: 45px; right: 0;
            background: #fff; border-radius: 8px; box-shadow: var(--shadow-lg);
            list-style: none; padding: 0.5rem 0;
            transform: scale(0.8); opacity: 0; visibility: hidden;
            transform-origin: bottom right; transition: all 0.2s ease; z-index: 20;
        }
        .menu-options.active { transform: scale(1); opacity: 1; visibility: visible; }
        .menu-options li { padding: 0.75rem 1.5rem; cursor: pointer; display: flex; align-items: center; gap: 10px; transition: background 0.2s; }
        .menu-options li:hover { background: #f1f5f9; }
        /* Modal Styles */
        .modal-overlay {
            position: fixed; top: 0; left: 0; width: 100%; height: 100%;
            background: rgba(30, 41, 59, 0.6);
            z-index: 1000;
            display: flex; justify-content: center; align-items: center;
            opacity: 0; visibility: hidden; transition: visibility 0.2s, opacity 0.2s;
        }
        .modal-overlay.active { opacity: 1; visibility: visible; }
        .modal-container {
            background: #fff; padding: 2rem; border-radius: 15px;
            width: 90%; max-width: 500px; box-shadow: var(--shadow-lg);
            transition: transform 0.3s ease; transform: scale(0.95);
        }
        .modal-overlay.active .modal-container { transform: scale(1); }
        .modal-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 1.5rem; }
        .modal-title { font-size: 1.5rem; }
        .close-modal-btn { background: none; border: none; font-size: 1.5rem; cursor: pointer; color: #9ca3af; }
        .modal-form .form-group { margin-bottom: 1.25rem; }
        .modal-form label { display: block; margin-bottom: 0.5rem; font-weight: 600; font-size: 0.9rem; }
        .modal-form input, .modal-form textarea {
            width: 100%; padding: 12px; border: 1px solid #d1d5db;
            border-radius: 8px; font-family: 'Inter', sans-serif; transition: border-color 0.2s, box-shadow 0.2s;
        }
        .modal-form input:focus, .modal-form textarea:focus { outline: none; border-color: var(--primary-color); box-shadow: 0 0 0 3px rgba(13, 148, 136, 0.2); }
        .modal-form textarea { resize: vertical; min-height: 100px; }
        .status-toggle { display: flex; align-items: center; gap: 10px; }
        .switch { position: relative; display: inline-block; width: 50px; height: 28px; }
        .switch input { opacity: 0; width: 0; height: 0; }
        .slider { position: absolute; cursor: pointer; top: 0; left: 0; right: 0; bottom: 0; background-color: #ccc; transition: .4s; border-radius: 34px; }
        .slider:before { position: absolute; content: ""; height: 20px; width: 20px; left: 4px; bottom: 4px; background-color: white; transition: .4s; border-radius: 50%; }
        input:checked + .slider { background-color: var(--status-active); }
        input:checked + .slider:before { transform: translateX(22px); }
        .modal-footer { text-align: right; margin-top: 2rem; }
        .modal-submit-btn {
            padding: 12px 30px; background: var(--primary-color); color: #fff;
            border: none; border-radius: 8px; cursor: pointer; font-weight: 600;
        }
    </style>
</head>
<body>
    <div class="page-container">
        <header class="page-header">
            <h1 class="header-title">Manage Properties</h1>
            <a href="add_property.php" class="add-property-btn"><i class="fas fa-plus"></i> Add New Property</a>
        </header>

        <main class="property-grid" id="property-grid">
            <?php if ($result && $result->num_rows > 0): ?>
                <?php while($row = $result->fetch_assoc()): ?>
                    <?php
                        $images = [];
                        if (!empty($row['image1'])) $images[] = $row['image1'];
                        if (!empty($row['image2'])) $images[] = $row['image2'];
                        if (!empty($row['image3'])) $images[] = $row['image3'];
                        if (!empty($row['image4'])) $images[] = $row['image4'];
                    ?>
                    <div class="property-card" data-property-id="<?php echo $row['id']; ?>">
                        <div class="card-slider-wrapper">
                            <div class="status-badge <?php echo $row['status']; ?>">
                                <?php echo $row['status'] === 'active' ? 'Available' : 'Booked'; ?>
                            </div>
                            <div class="card-slider">
                                <?php if (!empty($images)): ?>
                                    <div class="swiper">
                                        <div class="swiper-wrapper">
                                            <?php foreach ($images as $img_path): ?>
                                                <div class="swiper-slide"><img src="<?php echo htmlspecialchars($img_path); ?>" alt="Property Image"></div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="card-content">
                            <h3 class="card-header"><?php echo htmlspecialchars($row['header']); ?></h3>
                            <p class="card-property-type"><?php echo htmlspecialchars($row['property_type']); ?></p>
                            <div class="card-rate">
                                <i class="fas fa-indian-rupee-sign"></i>
                                <?php echo number_format($row['rate'], 2); ?>
                                <span>/ night</span>
                            </div>
                            <div class="detail-section">
                                <h4>Description</h4>
                                <p><?php echo nl2br(htmlspecialchars($row['description'])); ?></p>
                            </div>
                        </div>
                        <div class="card-footer">
                            <span><i class="fas fa-calendar-alt"></i> Added: <?php echo date("M j, Y", strtotime($row['upload_date'])); ?></span>
                             <div class="card-menu">
                                <button class="menu-toggle"><i class="fas fa-ellipsis-h"></i></button>
                                <ul class="menu-options">
                                    <li class="edit-btn"><i class="fas fa-edit"></i> Edit</li>
                                    <li class="delete-btn"><i class="fas fa-trash-alt"></i> Delete</li>
                                </ul>
                            </div>
                        </div>
                        <div class="property-data" style="display:none;"><?php echo json_encode($row); ?></div>
                    </div>
                <?php endwhile; ?>
            <?php endif; ?>
        </main>
    </div>

    <div class="modal-overlay" id="edit-modal">
        <div class="modal-container">
            <div class="modal-header">
                <h2 class="modal-title">Edit Property Details</h2>
                <button class="close-modal-btn"><i class="fas fa-times"></i></button>
            </div>
            <form id="edit-form" class="modal-form">
                <input type="hidden" name="id" id="edit-property-id">
                <input type="hidden" name="action" value="update">
                <div class="form-group">
                    <label for="edit-header">Header</label>
                    <input type="text" id="edit-header" name="header" required>
                </div>
                <div class="form-group">
                    <label for="edit-rate">Rate per Night (₹)</label>
                    <input type="number" id="edit-rate" name="rate" min="0" step="0.01" required>
                </div>
                <div class="form-group">
                    <label for="edit-description">Description</label>
                    <textarea id="edit-description" name="description"></textarea>
                </div>
                <div class="form-group">
                    <label for="edit-rules">Rules</label>
                    <textarea id="edit-rules" name="rules"></textarea>
                </div>
                <div class="form-group">
                    <label>Status</label>
                    <div class="status-toggle">
                        <span>Booked</span>
                        <label class="switch">
                            <input type="checkbox" id="edit-status" name="status" value="active">
                            <span class="slider"></span>
                        </label>
                        <span>Available</span>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="submit" class="modal-submit-btn">Save Changes</button>
                </div>
            </form>
        </div>
    </div>
    
    <div class="modal-overlay" id="delete-confirm-modal">
         <div class="modal-container" style="max-width: 400px; text-align: center;">
            <i class="fas fa-exclamation-triangle" style="font-size: 3rem; color: var(--status-inactive); margin-bottom: 1rem;"></i>
            <h2 class="modal-title">Are you sure?</h2>
            <p style="color: var(--light-text-color); margin: 1rem 0;">This action cannot be undone. All files and data will be permanently deleted.</p>
            <div id="delete-confirm-buttons" style="display: flex; justify-content: center; gap: 1rem; margin-top: 1.5rem;">
                <button class="cancel-delete-btn" style="padding: 10px 25px; border-radius: 8px; border: 1px solid #d1d5db; cursor: pointer; font-weight: 600; background: #fff;">Cancel</button>
                <button class="confirm-delete-btn" style="padding: 10px 25px; border-radius: 8px; border: none; cursor: pointer; font-weight: 600; background: var(--status-inactive); color: #fff;">Delete</button>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/swiper@11/swiper-bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', () => {
        
        document.querySelectorAll('.swiper').forEach(swiperEl => {
            new Swiper(swiperEl, { loop: true, effect: 'fade', fadeEffect: { crossFade: true } });
        });

        const editModal = document.getElementById('edit-modal');
        const deleteModal = document.getElementById('delete-confirm-modal');
        let propertyIdToDelete = null;

        const openModal = (modal) => modal.classList.add('active');
        const closeModal = (modal) => modal.classList.remove('active');

        document.getElementById('property-grid').addEventListener('click', e => {
            const card = e.target.closest('.property-card');
            if (!card) return;

            if (e.target.closest('.menu-toggle')) {
                card.querySelector('.menu-options').classList.toggle('active');
            }
            else if (e.target.closest('.edit-btn')) {
                const data = JSON.parse(card.querySelector('.property-data').textContent);
                document.getElementById('edit-property-id').value = data.id;
                document.getElementById('edit-header').value = data.header;
                document.getElementById('edit-rate').value = parseFloat(data.rate).toFixed(2); // Set rate value
                document.getElementById('edit-description').value = data.description;
                document.getElementById('edit-rules').value = data.rules;
                document.getElementById('edit-status').checked = (data.status === 'active');
                openModal(editModal);
            }
            else if (e.target.closest('.delete-btn')) {
                propertyIdToDelete = card.dataset.propertyId;
                openModal(deleteModal);
            }
        });

        document.querySelectorAll('.close-modal-btn, .cancel-delete-btn').forEach(btn => {
            btn.addEventListener('click', () => closeModal(btn.closest('.modal-overlay')));
        });

        document.getElementById('edit-form').addEventListener('submit', async e => {
            e.preventDefault();
            const form = e.target;
            const formData = new FormData(form);
            formData.set('status', document.getElementById('edit-status').checked ? 'active' : 'inactive');

            const response = await fetch(window.location.href, { method: 'POST', body: formData });
            const result = await response.json();

            if (result.success) {
                const card = document.querySelector(`.property-card[data-property-id="${formData.get('id')}"]`);
                const data = result.updatedData;
                // Update UI with new data
                card.querySelector('.card-header').textContent = data.header;
                card.querySelector('.card-rate').innerHTML = `<i class="fas fa-indian-rupee-sign"></i> ${parseFloat(data.rate).toLocaleString('en-IN', {minimumFractionDigits: 2, maximumFractionDigits: 2})} <span>/ night</span>`;
                card.querySelector('.detail-section p').textContent = data.description;
                const statusBadge = card.querySelector('.status-badge');
                statusBadge.className = `status-badge ${data.status}`;
                statusBadge.textContent = data.status === 'active' ? 'Available' : 'Booked';
                // Update hidden data
                const hiddenDataEl = card.querySelector('.property-data');
                const hiddenData = JSON.parse(hiddenDataEl.textContent);
                Object.assign(hiddenData, data, {id: formData.get('id')});
                hiddenDataEl.textContent = JSON.stringify(hiddenData);
                
                closeModal(editModal);
            } else {
                alert('Error: ' + result.message);
            }
        });

        document.querySelector('.confirm-delete-btn').addEventListener('click', async () => {
            if (!propertyIdToDelete) return;
            const formData = new FormData();
            formData.append('action', 'delete');
            formData.append('id', propertyIdToDelete);

            const response = await fetch(window.location.href, { method: 'POST', body: formData });
            const result = await response.json();

            if (result.success) {
                const cardToDelete = document.querySelector(`.property-card[data-property-id="${propertyIdToDelete}"]`);
                cardToDelete.style.transition = 'opacity 0.3s, transform 0.3s';
                cardToDelete.style.opacity = '0';
                cardToDelete.style.transform = 'scale(0.9)';
                setTimeout(() => cardToDelete.remove(), 300);
                closeModal(deleteModal);
            } else {
                alert('Error: ' + result.message);
            }
            propertyIdToDelete = null;
        });
    });
    </script>
</body>
</html>
<?php
$link->close();
?>