<?php
// --- SETUP & AUTOLOADING ---
// Use Composer's autoloader to load PHPMailer and FPDF
require 'vendor/autoload.php';

// Import PHPMailer classes into the global namespace
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

// --- .ENV FILE PARSING ---
// Simple function to parse the .env file
function parseEnv($path) {
    if (!file_exists($path)) {
        throw new \InvalidArgumentException(sprintf('%s does not exist', $path));
    }
    $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos(trim($line), '#') === 0) continue;
        list($name, $value) = explode('=', $line, 2);
        $name = trim($name);
        $value = trim($value);
        if (!array_key_exists($name, $_SERVER) && !array_key_exists($name, $_ENV)) {
            putenv(sprintf('%s=%s', $name, $value));
            $_ENV[$name] = $value;
            $_SERVER[$name] = $value;
        }
    }
}
// Load the .env file from the root of your project
parseEnv(__DIR__ . '/.env');

// --- HELPER FUNCTION FOR NIGHT CALCULATION ---
/**
 * Calculates the number of nights between two dates.
 * @param string $start_date The arrival date (e.g., '2025-08-12')
 * @param string $end_date The departure date (e.g., '2025-08-14')
 * @return int The total number of nights.
 */
function calculate_nights($start_date, $end_date) {
    $start = new DateTime($start_date);
    $end = new DateTime($end_date);
    $interval = $start->diff($end);
    return $interval->days;
}

// --- SESSION & SECURITY ---
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// --- DATABASE CONNECTION ---
require_once 'config/db.php'; 

// --- GET AND VALIDATE PROPERTY ID ---
$property_id = isset($_GET['id']) ? intval($_GET['id']) : 0;
if ($property_id === 0) {
    header("Location: index.php");
    exit();
}

// --- BOOKING LOGIC ---
$booking_message = '';
$booking_message_type = '';

if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['book_stay'])) {
    if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
        $booking_message = 'You must be logged in to book a stay.';
        $booking_message_type = 'error';
    } else {
        // --- FORM DATA ---
        $user_id = $_SESSION['userid'];
        $arrival_date = $_POST['arrival_date'];
        $departure_date = $_POST['departure_date'];
        $adults = intval($_POST['adults']);
        $children = intval($_POST['children']);
        $booking_date = date('Y-m-d H:i:s');
        $booking_id = 'FLORA' . mt_rand(1000, 9999);

        // --- VALIDATION ---
        if (empty($arrival_date) || empty($departure_date) || $departure_date <= $arrival_date) {
            $booking_message = 'Please select valid arrival and departure dates.';
            $booking_message_type = 'error';
        } else {
            // --- PRICE CALCULATION & DATA FETCHING ---
            // Fetch property rate FIRST to calculate total price
            $stmt_prop = $conn->prepare("SELECT header, location, rate FROM properties WHERE id = ?");
            $stmt_prop->bind_param("i", $property_id);
            $stmt_prop->execute();
            $prop_result = $stmt_prop->get_result()->fetch_assoc();
            $stmt_prop->close();

            if (!$prop_result) {
                $booking_message = 'Error: Property details could not be found.';
                $booking_message_type = 'error';
            } else {
                // Calculate total price
                $number_of_nights = calculate_nights($arrival_date, $departure_date);
                $total_price = $number_of_nights * $prop_result['rate'];

                // NOTE: Assumes you have added a 'total_price' column (e.g., DECIMAL(10, 2)) to your 'bookings' table.
                $stmt_booking = $conn->prepare("INSERT INTO bookings (booking_id, property_id, user_id, arrival_date, departure_date, adults, children, booking_date, total_price) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt_booking->bind_param("siisssisd", $booking_id, $property_id, $user_id, $arrival_date, $departure_date, $adults, $children, $booking_date, $total_price);

                if ($stmt_booking->execute()) {
                    $booking_message = "Booking Confirmed! A confirmation email with your invoice (ID: $booking_id) has been sent.";
                    $booking_message_type = 'success';

                    // --- FETCH USER DETAILS FOR EMAIL/PDF ---
                    $stmt_user = $conn->prepare("SELECT name, email FROM users WHERE id = ?");
                    $stmt_user->bind_param("i", $user_id);
                    $stmt_user->execute();
                    $user_result = $stmt_user->get_result()->fetch_assoc();
                    $stmt_user->close();

                    // --- PDF GENERATION (FPDF) ---
                    $pdf = new FPDF();
                    $pdf->AddPage();
                    $pdf->SetFont('Arial', 'B', 16);
                    if (file_exists('logo.png')) {
                        $pdf->Image('logo.png', 10, 6, 30);
                    }
                    $pdf->Cell(0, 10, 'Booking Invoice', 0, 1, 'C');
                    $pdf->Ln(10);
                    $pdf->SetFont('Arial', '', 12);
                    $pdf->Cell(0, 10, 'Thank you for choosing FLORACION Hospitality!', 0, 1);
                    $pdf->Cell(0, 10, "Booking ID: " . $booking_id, 0, 1);
                    $pdf->Ln(5);
                    // Table Header
                    $pdf->SetFont('Arial', 'B', 12);
                    $pdf->Cell(50, 10, 'Item', 1);
                    $pdf->Cell(0, 10, 'Details', 1);
                    $pdf->Ln();
                    // Table Rows
                    $pdf->SetFont('Arial', '', 12);
                    $pdf->Cell(50, 10, 'Guest Name', 1); $pdf->Cell(0, 10, $user_result['name'], 1); $pdf->Ln();
                    $pdf->Cell(50, 10, 'Property', 1); $pdf->Cell(0, 10, $prop_result['header'], 1); $pdf->Ln();
                    $pdf->Cell(50, 10, 'Location', 1); $pdf->Cell(0, 10, $prop_result['location'], 1); $pdf->Ln();
                    $pdf->Cell(50, 10, 'Arrival', 1); $pdf->Cell(0, 10, $arrival_date, 1); $pdf->Ln();
                    $pdf->Cell(50, 10, 'Departure', 1); $pdf->Cell(0, 10, $departure_date, 1); $pdf->Ln();
                    $pdf->Cell(50, 10, 'Rate per Night', 1); $pdf->Cell(0, 10, 'Rs. ' . number_format($prop_result['rate']), 1); $pdf->Ln();
                    $pdf->Cell(50, 10, 'Number of Nights', 1); $pdf->Cell(0, 10, $number_of_nights, 1); $pdf->Ln();
                    $pdf->SetFont('Arial', 'B', 12); // Bold for Total
                    $pdf->Cell(50, 10, 'Total Amount', 1); $pdf->Cell(0, 10, 'Rs. ' . number_format($total_price), 1); $pdf->Ln();

                    $pdf_path = sys_get_temp_dir() . '/booking_' . $booking_id . '.pdf';
                    $pdf->Output('F', $pdf_path);

                    // --- EMAIL SENDING (PHPMailer) ---
                    $mail = new PHPMailer(true);
                    try {
                        //Server settings
                        $mail->isSMTP();
                        $mail->Host       = $_ENV['MAIL_HOST'];
                        $mail->SMTPAuth   = true;
                        $mail->Username   = $_ENV['MAIL_USERNAME'];
                        $mail->Password   = $_ENV['MAIL_PASSWORD'];
                        $mail->SMTPSecure = $_ENV['MAIL_ENCRYPTION'];
                        $mail->Port       = $_ENV['MAIL_PORT'];

                        //Recipients
                        $mail->setFrom($_ENV['MAIL_USERNAME'], $_ENV['MAIL_FROM_NAME']);
                        $mail->addAddress($user_result['email'], $user_result['name']);
                        $mail->addAttachment($pdf_path);

                        //Content
                        $mail->isHTML(true);
                        $mail->Subject = 'Your Booking Confirmation from FLORACION Hospitality (ID: ' . $booking_id . ')';
                        $mail->Body    = "
                            <div style='font-family: Arial, sans-serif; line-height: 1.6;'>
                                <h2>Booking Confirmed!</h2>
                                <p>Dear " . htmlspecialchars($user_result['name']) . ",</p>
                                <p>Your stay at <strong>" . htmlspecialchars($prop_result['header']) . "</strong> has been confirmed. We are excited to host you!</p>
                                <p>Your booking invoice is attached. Please review the summary below:</p>
                                <table style='width: 100%; border-collapse: collapse;'>
                                    <tr style='background-color: #f2f2f2;'><td style='padding: 8px; border: 1px solid #ddd;'>Booking ID</td><td style='padding: 8px; border: 1px solid #ddd;'><strong>" . $booking_id . "</strong></td></tr>
                                    <tr><td style='padding: 8px; border: 1px solid #ddd;'>Arrival Date</td><td style='padding: 8px; border: 1px solid #ddd;'>" . $arrival_date . "</td></tr>
                                    <tr><td style='padding: 8px; border: 1px solid #ddd;'>Departure Date</td><td style='padding: 8px; border: 1px solid #ddd;'>" . $departure_date . "</td></tr>
                                    <tr><td style='padding: 8px; border: 1px solid #ddd;'>Number of Nights</td><td style='padding: 8px; border: 1px solid #ddd;'>" . $number_of_nights . "</td></tr>
                                    <tr style='background-color: #f2f2f2;'><td style='padding: 8px; border: 1px solid #ddd;'><strong>Total Amount</strong></td><td style='padding: 8px; border: 1px solid #ddd;'><strong>Rs. " . number_format($total_price) . "</strong></td></tr>
                                </table>
                                <p>We look forward to welcoming you.</p>
                                <p>Best regards,<br>The Team at FLORACION Hospitality</p>
                            </div>
                        ";

                        $mail->send();
                    } catch (Exception $e) {
                        // Optional: Log email sending errors if needed
                    }
                } else {
                    $booking_message = 'Error: Could not process your booking. Please try again.';
                    $booking_message_type = 'error';
                }
                $stmt_booking->close();
            }
        }
    }
}

// --- FETCH PROPERTY DETAILS FOR PAGE DISPLAY ---
$stmt_prop_details = $conn->prepare("SELECT * FROM properties WHERE id = ? AND status = 'active'");
$stmt_prop_details->bind_param("i", $property_id);
$stmt_prop_details->execute();
$result = $stmt_prop_details->get_result();
if ($result->num_rows === 0) { die("Property not found."); }
$property = $result->fetch_assoc();
$stmt_prop_details->close();
$images = [];
for ($i = 1; $i <= 4; $i++) { if (!empty($property["image$i"])) { $images[] = $property["image$i"]; } }
if (empty($images)) { $images[] = 'https://placehold.co/1200x800/1e293b/ffffff?text=Image+Unavailable'; }

// --- FETCH SUGGESTIONS ---
$suggestions = [];
$stmt_suggestions = $conn->prepare("SELECT id, header, image1, location, rate FROM properties WHERE (location = ? OR property_type = ?) AND id != ? AND status = 'active' LIMIT 4");
$stmt_suggestions->bind_param("ssi", $property['location'], $property['property_type'], $property_id);
$stmt_suggestions->execute();
$result_suggestions = $stmt_suggestions->get_result();
while ($row = $result_suggestions->fetch_assoc()) { $suggestions[] = $row; }
$stmt_suggestions->close();
$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($property['header']); ?> - Floracion Hospitality</title>
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Playfair+Display:wght@700;800&family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">

    <style>
        body { font-family: 'Poppins', sans-serif; background-color: #ffffff; color: #111827; overflow-x: hidden; }
        .font-serif { font-family: 'Playfair Display', serif; }
        .hero-section { height: 45vh; min-height: 350px; }
        .gallery-main-image { transition: transform 0.4s cubic-bezier(0.25, 0.46, 0.45, 0.94), opacity 0.4s ease-in-out; }
        .gallery-thumbnail { transition: all 0.3s ease; }
        .gallery-thumbnail.active { border-color: #059669; transform: scale(1.05); box-shadow: 0 4px 15px rgba(5, 150, 105, 0.3); }
        .btn-primary { background: linear-gradient(45deg, #10b981, #059669); transition: all 0.3s ease; box-shadow: 0 4px 15px rgba(16, 185, 129, 0.2); }
        .btn-primary:hover { transform: translateY(-3px) scale(1.02); box-shadow: 0 7px 20px rgba(16, 185, 129, 0.3); }
        .date-input-icon::-webkit-calendar-picker-indicator { background: transparent; bottom: 0; color: transparent; cursor: pointer; height: auto; left: 0; position: absolute; right: 0; top: 0; width: auto; }
        .gsap-reveal { opacity: 0; transform: translateY(40px); }
    </style>
</head>
<body>

    <?php include 'navbar.php'; ?>

    <main class="main-container">
        <section class="hero-section relative bg-gray-900 flex items-center justify-center text-white overflow-hidden">
            <div class="absolute inset-0 w-full h-full z-0"><img src="owner//<?php echo $images[0]; ?>" class="absolute inset-0 w-full h-full object-cover opacity-30" alt="Background"><div class="absolute inset-0 bg-gradient-to-t from-black/70 to-transparent"></div></div>
            <div class="container mx-auto px-6 relative text-center z-10"><div class="hero-content"><span class="hero-subtitle block text-emerald-400 font-semibold uppercase tracking-widest gsap-reveal"><?php echo htmlspecialchars($property['property_type']); ?></span><h1 class="hero-title font-serif text-5xl md:text-7xl mt-2 mb-4 gsap-reveal"><?php echo htmlspecialchars($property['header']); ?></h1><p class="hero-location text-lg text-gray-200 flex items-center justify-center gap-2 gsap-reveal"><i class="fas fa-map-marker-alt"></i><span><?php echo htmlspecialchars($property['location']); ?></span></p></div></div>
        </section>

        <div class="page-wrapper container mx-auto px-6 py-16 lg:py-24">
            <div class="grid lg:grid-cols-5 gap-12 xl:gap-16">
                <div class="lg:col-span-3">
                    <section class="gallery-section mb-12" data-aos="fade-up">
                        <div class="gallery-main-image-wrapper rounded-2xl overflow-hidden shadow-2xl mb-4 relative"><img id="gallery-main-image" src="owner//<?php echo $images[0]; ?>" alt="Main view of <?php echo htmlspecialchars($property['header']); ?>" class="w-full h-[350px] md:h-[550px] object-cover"></div>
                        <?php if (count($images) > 1): ?><div class="gallery-thumbnails-wrapper grid grid-cols-2 md:grid-cols-4 gap-4"><?php foreach ($images as $index => $img): ?><div class="gallery-thumbnail-container"><img src="owner//<?php echo $img; ?>" alt="Thumbnail <?php echo $index + 1; ?>" class="gallery-thumbnail w-full h-24 md:h-32 object-cover rounded-lg cursor-pointer border-4 border-transparent <?php echo $index == 0 ? 'active' : ''; ?>"></div><?php endforeach; ?></div><?php endif; ?>
                    </section>
                    <section class="info-section" data-aos="fade-up" data-aos-delay="200">
                        <div class="info-header flex items-center gap-4 border-b border-gray-200 pb-4 mb-6"><i class="fas fa-home text-3xl text-emerald-600"></i><h2 class="font-serif text-4xl text-gray-900">About this Stay</h2></div>
                        <p class="description text-gray-600 leading-relaxed whitespace-pre-wrap"><?php echo htmlspecialchars($property['description']); ?></p>
                        <div class="amenities mt-12"><div class="amenities-header flex items-center gap-4 border-b border-gray-200 pb-4 mb-6"><i class="fas fa-concierge-bell text-3xl text-emerald-600"></i><h3 class="font-serif text-3xl text-gray-900">What This Place Offers</h3></div><ul class="amenities-list grid grid-cols-1 sm:grid-cols-2 gap-x-8 gap-y-4 text-gray-700"><?php $rules_list = explode("\n", $property['rules']); foreach ($rules_list as $rule): if(!empty(trim($rule))): ?><li class="amenity-item flex items-center" data-aos="fade-left" data-aos-offset="100"><i class="fas fa-check-circle text-emerald-500 mr-3 text-lg"></i><?php echo htmlspecialchars(trim($rule)); ?></li><?php endif; endforeach; ?></ul></div>
                    </section>
                </div>

                <div class="lg:col-span-2">
                    <aside class="booking-widget-wrapper lg:sticky top-24" data-aos="fade-up" data-aos-delay="300">
                        <div class="booking-widget bg-white rounded-2xl p-8 shadow-2xl border border-gray-200">
                            <div class="booking-price-container flex justify-between items-center border-b border-gray-200 pb-6 mb-6">
                                <p class="font-serif text-4xl text-emerald-600">₹<?php echo number_format($property['rate']); ?></p>
                                <p class="text-gray-500 text-lg">per night</p>
                            </div>

                            <?php if (isset($_SESSION['loggedin']) && $_SESSION['loggedin'] === true): ?>
                                <?php if ($booking_message): ?>
                                    <div class="booking-message p-4 mb-4 text-sm rounded-lg <?php echo ($booking_message_type == 'success') ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>" role="alert">
                                        <?php echo $booking_message; ?>
                                    </div>
                                    
                                    <?php if ($booking_message_type == 'success'): ?>
                                        <script>
                                            setTimeout(function() {
                                                window.location.href = 'index.php';
                                            }, 2000); // 2000 milliseconds = 2 seconds
                                        </script>
                                    <?php endif; ?>
                                <?php endif; ?>
                                
                                <form class="booking-form space-y-4" action="property_details.php?id=<?php echo $property_id; ?>" method="POST">
                                    <input type="hidden" name="book_stay" value="1">
                                    <div class="relative form-group"><label for="arrival_date" class="block text-sm font-medium text-gray-700 mb-1">Arrival</label><i class="fas fa-calendar-alt absolute left-3 top-10 text-gray-400 z-10"></i><input type="date" id="arrival_date" name="arrival_date" required class="w-full bg-gray-100 border-gray-300 text-gray-900 rounded-lg p-3 pl-10 focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500 transition date-input-icon"></div>
                                    <div class="relative form-group"><label for="departure_date" class="block text-sm font-medium text-gray-700 mb-1">Departure</label><i class="fas fa-calendar-alt absolute left-3 top-10 text-gray-400 z-10"></i><input type="date" id="departure_date" name="departure_date" required class="w-full bg-gray-100 border-gray-300 text-gray-900 rounded-lg p-3 pl-10 focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500 transition date-input-icon"></div>
                                    <div class="grid grid-cols-2 gap-4"><div class="form-group"><label for="adults" class="block text-sm font-medium text-gray-700 mb-1">Adults</label><select id="adults" name="adults" class="w-full bg-gray-100 border-gray-300 text-gray-900 rounded-lg p-3 focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500 transition"><option>1</option><option selected>2</option><option>3</option><option>4</option></select></div><div class="form-group"><label for="children" class="block text-sm font-medium text-gray-700 mb-1">Children</label><select id="children" name="children" class="w-full bg-gray-100 border-gray-300 text-gray-900 rounded-lg p-3 focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500 transition"><option selected>0</option><option>1</option><option>2</option><option>3</option></select></div></div>
                                    <button type="submit" class="w-full btn-primary text-white font-bold py-4 rounded-lg text-lg"><i class="fas fa-calendar-check mr-2"></i> Book Your Stay</button>
                                </form>
                            <?php else: ?>
                                <div class="login-prompt text-center p-4">
                                    <div id="login-prompt-icon" class="relative w-24 h-24 mx-auto mb-4">
                                        <svg class="w-full h-full text-emerald-500" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M8 10V7C8 4.79086 9.79086 3 12 3C14.2091 3 16 4.79086 16 7V10" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/><path d="M5 10H19V20C19 20.5523 18.5523 21 18 21H6C5.44772 21 5 20.5523 5 20V10Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/><path id="keyhole" d="M12 14V16" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>
                                    </div>
                                    <h3 class="font-serif text-2xl font-bold text-gray-800">Whoa there, traveler!</h3>
                                    <p class="text-gray-500 mt-2 mb-6">Looks like you need a key to book this room. Login first!</p>
                                    <a href="login.php" class="w-full inline-block btn-primary text-white font-bold py-3 px-6 rounded-lg text-lg"><i class="fas fa-key mr-2"></i> Login or Register</a>
                                </div>
                            <?php endif; ?>
                        </div>
                    </aside>
                </div>
            </div>
        </div>
        
        <?php if (!empty($suggestions)): ?>
        <section class="suggestions-section bg-gray-50 py-16 lg:py-24">
            <div class="container mx-auto px-6"><div class="suggestions-header text-center mb-12" data-aos="fade-up"><h2 class="font-serif text-4xl text-gray-900">You Might Also Like</h2><p class="text-gray-500 mt-2">Other properties you may be interested in.</p></div><div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-8"><?php foreach ($suggestions as $index => $suggestion): ?><a href="property_details.php?id=<?php echo $suggestion['id']; ?>" class="suggestion-card block bg-white rounded-xl shadow-md overflow-hidden transition-all duration-300 hover:shadow-xl hover:-translate-y-2" data-aos="fade-up" data-aos-delay="<?php echo $index * 100; ?>"><img src="owner//<?php echo htmlspecialchars($suggestion['image1']); ?>" alt="<?php echo htmlspecialchars($suggestion['header']); ?>" class="w-full h-48 object-cover"><div class="p-5"><h4 class="font-bold text-lg text-gray-800 truncate"><?php echo htmlspecialchars($suggestion['header']); ?></h4><p class="text-sm text-gray-500 flex items-center gap-2 mt-1"><i class="fas fa-map-marker-alt"></i><?php echo htmlspecialchars($suggestion['location']); ?></p><p class="text-emerald-600 font-bold text-md mt-3">₹<?php echo number_format($suggestion['rate']); ?> / night</p></div></a><?php endforeach; ?></div></div>
        </section>
        <?php endif; ?>
    </main>

    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/gsap/3.12.5/gsap.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', () => {
            AOS.init({ once: true, duration: 800, easing: 'ease-in-out-quad' });
            
            const tl = gsap.timeline({ defaults: { ease: 'power3.out' } });
            tl.staggerFromTo('.gsap-reveal', 1, { opacity: 0, y: 40 }, { opacity: 1, y: 0 }, 0.2)
              .from('.booking-widget', { opacity: 0, y: 50, duration: 1 }, "-=0.8");
            
            if (document.querySelector('.booking-form')) {
                tl.staggerFromTo('.booking-form .form-group', 0.5, { opacity: 0, y: 20 }, { opacity: 1, y: 0 }, 0.1, "-=0.7");
            } else if (document.querySelector('.login-prompt')) {
                tl.from('.login-prompt', { opacity: 0, scale: 0.9, duration: 0.5 }, "-=0.5");
                gsap.to('#login-prompt-icon', { y: -10, repeat: -1, yoyo: true, duration: 1.5, ease: 'power1.inOut' });
                gsap.to('#keyhole', { scale: 1.5, transformOrigin: 'center', repeat: -1, yoyo: true, duration: 3, ease: 'power1.inOut' });
            }
            
            const mainImage = document.getElementById('gallery-main-image');
            const thumbnails = document.querySelectorAll('.gallery-thumbnail');
            thumbnails.forEach(thumb => {
                thumb.addEventListener('click', function() {
                    if (this.classList.contains('active')) return;
                    const newSrc = this.src;
                    gsap.to(mainImage, {
                        opacity: 0, scale: 1.05, duration: 0.3, ease: 'power2.in',
                        onComplete: () => {
                            mainImage.src = newSrc;
                            gsap.to(mainImage, { opacity: 1, scale: 1, duration: 0.4, ease: 'power2.out' });
                        }
                    });
                    thumbnails.forEach(t => t.classList.remove('active'));
                    this.classList.add('active');
                });
            });
            
            const arrivalInput = document.getElementById('arrival_date');
            const departureInput = document.getElementById('departure_date');
            if (arrivalInput && departureInput) {
                const today = new Date();
                today.setDate(today.getDate());
                const minDate = today.toISOString().split('T')[0];
                
                arrivalInput.setAttribute('min', minDate);
                
                arrivalInput.addEventListener('change', () => {
                    if (!arrivalInput.value) return;
                    let arrivalDate = new Date(arrivalInput.value);
                    arrivalDate.setDate(arrivalDate.getDate() + 1); // Departure must be at least one day after arrival
                    let minDeparture = arrivalDate.toISOString().split('T')[0];
                    departureInput.setAttribute('min', minDeparture);
                    
                    if (departureInput.value && new Date(departureInput.value) <= new Date(arrivalInput.value)) {
                        departureInput.value = '';
                    }
                });
            }
        });
    </script>

</body>
</html>