<?php
include 'config/db.php'; // Include the database connection file

$userInput = isset($_GET['location']) ? trim($_GET['location']) : '';
$properties = [];
$correctedLocation = '';

if (!empty($userInput)) {
    // --- FUZZY SEARCH LOGIC ---
    
    // 1. Get all unique locations from the database
    $locations_sql = "SELECT DISTINCT location FROM properties";
    $locations_result = $conn->query($locations_sql);
    $all_locations = [];
    if ($locations_result->num_rows > 0) {
        while ($row = $locations_result->fetch_assoc()) {
            $all_locations[] = $row['location'];
        }
    }

    // 2. Find the best match using Levenshtein distance
    $shortest = -1;
    $closest = '';
    foreach ($all_locations as $location) {
        $lev = levenshtein(strtolower($userInput), strtolower($location));
        if ($lev == 0) { // Perfect match
            $closest = $location;
            $shortest = 0;
            break;
        }
        if ($lev <= $shortest || $shortest < 0) {
            $closest = $location;
            $shortest = $lev;
        }
    }

    // 3. Decide if the match is good enough (threshold of 3)
    if ($shortest > -1 && $shortest < 3) {
        $correctedLocation = $closest;
    } else {
        $correctedLocation = $userInput; // If no good match, use original input
    }
    // --- END OF FUZZY SEARCH LOGIC ---

    // 4. Fetch properties based on the (potentially corrected) location
    $sql = "SELECT `id`, `property_type`, `rate`, `location`, `header`, `description`, `image1`, `image2`, `image3`, `image4` FROM `properties` WHERE `location` LIKE ?";
    $stmt = $conn->prepare($sql);
    if ($stmt) {
        $search_param = "%" . $correctedLocation . "%";
        $stmt->bind_param('s', $search_param);
        $stmt->execute();
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $properties[] = $row;
        }
        $stmt->close();
    }
} else {
    // If no location is provided, show all properties
    $sql = "SELECT `id`, `property_type`, `rate`, `location`, `header`, `description`, `image1`, `image2`, `image3`, `image4` FROM `properties`";
    $result = $conn->query($sql);
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $properties[] = $row;
        }
    }
}

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Search Results - Floracion</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Playfair+Display:wght@700&family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Poppins', sans-serif; }
        .font-serif { font-family: 'Playfair Display', serif; }
        .search-slider-image { transition: opacity 0.5s ease-in-out; }
        .search-slider-dot.active { background-color: white; transform: scale(1.2); }
    </style>
</head>
<body class="bg-gray-50">

    <?php include 'navbar.php'; ?>

    <main class="search-main-container">
        <div class="search-hero-section bg-gray-900 pt-32 pb-16 text-white text-center">
            <div class="container mx-auto px-4">
                <i class="fas fa-search text-4xl text-emerald-400 mb-4"></i>
                <h1 class="search-title text-4xl md:text-5xl font-serif">Search Results</h1>
                <p class="search-subtitle text-lg text-gray-300 mt-2">
                    <?php
                    if (!empty($userInput)) {
                        if (strtolower($userInput) != strtolower($correctedLocation)) {
                            echo 'Showing results for <strong class="text-emerald-400">' . htmlspecialchars($correctedLocation) . '</strong> (searched for "' . htmlspecialchars($userInput) . '")';
                        } else {
                            echo 'Showing properties in <strong class="text-emerald-400">' . htmlspecialchars($correctedLocation) . '</strong>';
                        }
                    } else {
                        echo 'Showing all our available properties';
                    }
                    ?>
                </p>
            </div>
        </div>

        <div class="search-results-container container mx-auto px-4 sm:px-6 py-16">
            <?php if (!empty($properties)): ?>
                <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-8">
                    <?php foreach ($properties as $property): ?>
                        <?php
                            $images = [];
                            for ($i = 1; $i <= 4; $i++) {
                                if (!empty($property["image$i"])) { $images[] = $property["image$i"]; }
                            }
                        ?>
                        <div class="search-property-card bg-white rounded-xl shadow-lg overflow-hidden flex flex-col transition-all duration-300 hover:shadow-2xl hover:-translate-y-2">
                            <div class="search-slider-container relative w-full h-56 group" data-property-id="<?php echo $property['id']; ?>">
                                <img class="search-slider-image w-full h-full object-cover" src="owner/<?php echo !empty($images) ? htmlspecialchars($images[0]) : 'https://placehold.co/600x400/a0aec0/ffffff?text=No+Image'; ?>" alt="<?php echo htmlspecialchars($property['header']); ?>">
                                <?php if(count($images) > 1): ?>
                                    <button class="search-slider-prev absolute top-1/2 left-2 -translate-y-1/2 bg-black/40 text-white rounded-full h-8 w-8 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity"><i class="fas fa-chevron-left"></i></button>
                                    <button class="search-slider-next absolute top-1/2 right-2 -translate-y-1/2 bg-black/40 text-white rounded-full h-8 w-8 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity"><i class="fas fa-chevron-right"></i></button>
                                    <div class="search-slider-dots-container absolute bottom-3 left-1/2 -translate-x-1/2 flex space-x-2">
                                        <?php foreach ($images as $index => $image): ?>
                                            <div class="search-slider-dot h-2 w-2 rounded-full bg-white/60 transition-all duration-300 <?php echo $index == 0 ? 'active' : ''; ?>"></div>
                                        <?php endforeach; ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="search-details-wrapper p-5 flex-grow flex flex-col">
                                <span class="search-property-type-badge text-xs font-semibold bg-emerald-100 text-emerald-800 px-2 py-1 rounded-full self-start mb-2"><?php echo htmlspecialchars($property['property_type']); ?></span>
                                <h3 class="search-property-header text-lg font-bold text-gray-800 mb-1 truncate"><?php echo htmlspecialchars($property['header']); ?></h3>
                                <p class="search-property-location text-sm text-gray-500 mb-3 flex items-center"><i class="fas fa-map-marker-alt mr-2 text-gray-400"></i><?php echo htmlspecialchars($property['location']); ?></p>
                                <p class="search-property-description text-sm text-gray-600 flex-grow mb-4"><?php echo substr(htmlspecialchars($property['description']), 0, 90) . (strlen($property['description']) > 90 ? '...' : ''); ?></p>
                                <div class="search-card-footer mt-auto pt-4 border-t border-gray-200 flex justify-between items-center">
                                    <p class="search-property-rate text-lg font-bold text-emerald-600">₹<?php echo number_format($property['rate']); ?><span class="text-sm font-normal text-gray-500">/night</span></p>
                                    <a href="property_details.php?id=<?php echo $property['id']; ?>" class="search-details-button px-4 py-2 bg-gray-800 text-white text-sm font-semibold rounded-lg hover:bg-gray-900 transition-colors">View</a>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <div class="search-no-results-container text-center py-16 px-6 bg-white rounded-xl shadow-md">
                    <div class="mx-auto flex items-center justify-center h-16 w-16 rounded-full bg-yellow-100"><i class="fas fa-hotel text-3xl text-yellow-500"></i></div>
                    <h3 class="mt-5 text-2xl font-semibold text-gray-800">No Properties Found</h3>
                    <p class="mt-2 text-md text-gray-500">We couldn't find any properties matching "<?php echo htmlspecialchars($userInput); ?>".<br>Try a different location or browse all properties.</p>
                    <a href="search.php" class="mt-6 inline-block px-6 py-3 bg-gray-800 text-white font-semibold rounded-lg hover:bg-gray-900 transition-colors">Browse All Properties</a>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <?php include 'footer.php'; ?>

<script>
document.addEventListener('DOMContentLoaded', () => {
    const imagesByProperty = <?php
        $jsImages = [];
        foreach ($properties as $p) {
            $jsImages[$p['id']] = [];
            for ($i = 1; $i <= 4; $i++) {
                if (!empty($p["image$i"])) { $jsImages[$p['id']][] = $p["image$i"]; }
            }
        }
        echo json_encode($jsImages);
    ?>;

    document.querySelectorAll('.search-slider-container').forEach(slider => {
        const propertyId = slider.dataset.propertyId;
        const images = imagesByProperty[propertyId] || [];
        if (images.length <= 1) return;

        const imageElement = slider.querySelector('.search-slider-image');
        const prevButton = slider.querySelector('.search-slider-prev');
        const nextButton = slider.querySelector('.search-slider-next');
        const dots = slider.querySelectorAll('.search-slider-dot');
        let currentIndex = 0;

        function updateSlider(newIndex) {
            currentIndex = newIndex;
            imageElement.style.opacity = '0';
            setTimeout(() => {
                imageElement.src = `owner/${images[currentIndex]}`;
                imageElement.style.opacity = '1';
                dots.forEach((dot, index) => dot.classList.toggle('active', index === currentIndex));
            }, 250);
        }

        prevButton.addEventListener('click', (e) => { e.preventDefault(); updateSlider((currentIndex - 1 + images.length) % images.length); });
        nextButton.addEventListener('click', (e) => { e.preventDefault(); updateSlider((currentIndex + 1) % images.length); });
    });
});
</script>

</body>
</html>
